/**
  *  \file ui/widgets/tabbar.cpp
  *  \brief Class ui::widgets::TabBar
  */

#include "ui/widgets/tabbar.hpp"
#include "gfx/complex.hpp"
#include "gfx/context.hpp"
#include "util/key.hpp"

struct ui::widgets::TabBar::TabInfo {
    size_t id;
    String_t name;
    util::Key_t key;

    TabInfo(size_t id, const String_t& name, util::Key_t key)
        : id(id), name(name), key(key)
        { }
};


ui::widgets::TabBar::TabBar(Root& root)
    : IconBox(root),
      m_root(root),
      m_tabs(),
      m_font(gfx::FontRequest().addSize(1))
{
    // ex UICardGroupTab::UICardGroupTab
}

ui::widgets::TabBar::~TabBar()
{
    // ex UICardGroupTab::~UICardGroupTab
}

void
ui::widgets::TabBar::addPage(size_t id, const String_t& name, util::Key_t key)
{
    // ex UICardGroupTab::addTab
    m_tabs.pushBackNew(new TabInfo(id, name, key));
    requestRedraw();
}

void
ui::widgets::TabBar::addPage(size_t id, const util::KeyString& name)
{
    // ex UICardGroupTab::addTab
    addPage(id, name.getString(), name.getKey());
}

void
ui::widgets::TabBar::setFont(gfx::FontRequest font)
{
    m_font = font;
}

size_t
ui::widgets::TabBar::getCurrentTabId() const
{
    size_t index = getCurrentItem();
    if (index < m_tabs.size()) {
        return m_tabs[index]->id;
    } else {
        return 0;
    }
}

void
ui::widgets::TabBar::setCurrentTabId(size_t id)
{
    for (size_t i = 0, n = m_tabs.size(); i < n; ++i) {
        const TabInfo* pTab = m_tabs[i];
        if (pTab->id == id) {
            setCurrentItem(i);
            break;
        }
    }
}

ui::layout::Info
ui::widgets::TabBar::getLayoutInfo() const
{
    // ex UICardGroupTab::getLayoutInfo
    afl::base::Ref<gfx::Font> font = m_root.provider().getFont(m_font);

    int min_x = 0;
    for (size_t i = 0; i < m_tabs.size(); ++i) {
        min_x += font->getTextWidth(m_tabs[i]->name) + 36;
    }

    int height = font->getLineHeight() + 3;

    return ui::layout::Info(gfx::Point(min_x, height), ui::layout::Info::GrowHorizontal);
}

int
ui::widgets::TabBar::getItemWidth(size_t nr) const
{
    if (nr < m_tabs.size()) {
        afl::base::Ref<gfx::Font> font = m_root.provider().getFont(m_font);
        return font->getTextWidth(m_tabs[nr]->name) + 36;
    } else {
        return 0;
    }
}

bool
ui::widgets::TabBar::isItemKey(size_t nr, util::Key_t key) const
{
    if (nr < m_tabs.size()) {
        const TabInfo* pTab = m_tabs[nr];
        return (pTab->key == key || pTab->key == (key & ~util::KeyMod_Alt));
    } else {
        return false;
    }
}

size_t
ui::widgets::TabBar::getNumItems() const
{
    return m_tabs.size();
}

void
ui::widgets::TabBar::drawItem(gfx::Canvas& can, gfx::Rectangle area, size_t item, ItemState state)
{
    // ex UICardGroupTab::drawContent
    afl::base::Ref<gfx::Font> font = m_root.provider().getFont(m_font);

    // Straight-forward port, could be more idiomatic
    int x = area.getLeftX();
    int top = area.getTopY();
    int bot = area.getBottomY();

    gfx::Context<util::SkinColor::Color> ctx(can, getColorScheme());  // used to draw interior of tabs which is visually inside the group
    ctx.useFont(*font);

    gfx::ColorQuad_t whiteColor = m_root.colorScheme().getColor(ui::Color_White);
    gfx::ColorQuad_t grayColor  = m_root.colorScheme().getColor(ui::Color_Gray);
    gfx::ColorQuad_t blackColor = m_root.colorScheme().getColor(ui::Color_Black);

    if (item < m_tabs.size()) {
        // Status
        const TabInfo& tab = *m_tabs[item];
        const bool isCurrent = (state == Selected);

        // first, draw lines at bottom
        int this_width = font->getTextWidth(tab.name) + 20;
        int lwidth = isCurrent ? 10 : this_width+16;
        can.drawHLine(gfx::Point(x, bot-2), lwidth, whiteColor, gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
        can.drawHLine(gfx::Point(x, bot-1), lwidth, grayColor,  gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
        if (isCurrent) {
            drawBackground(ctx, gfx::Rectangle(x + 10, bot - 2, this_width + 6, 2));
        }

        // left side of tab
        x += 10;
        can.drawVLine(gfx::Point(x-1, top), bot-2-top, whiteColor, gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
        can.drawVLine(gfx::Point(x, top+1), bot-2-top, grayColor,  gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);

        // top
        can.drawHLine(gfx::Point(x, top),   this_width, whiteColor, gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
        can.drawHLine(gfx::Point(x, top+1), this_width, grayColor,  gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);

        // content and rhs
        int cx = 0, he = bot - top - 2, lastl = 1, bx = x + this_width;
        for (int l = 1; l < he; ++l) {
            int w = 6 * l / he;
            if (w != cx) {
                can.drawBar(gfx::Rectangle(bx + cx, top + lastl, 2, l - lastl), blackColor, blackColor, gfx::FillPattern::SOLID, gfx::OPAQUE_ALPHA);
                if (lastl == 1 && l != 1) { // HACK HACK HACK
                    ++lastl;
                }
                drawBackground(ctx, gfx::Rectangle(x+1, top + lastl, cx+this_width-1, l - lastl));
                lastl = l;
                cx = w;
            }
        }
        drawBackground(ctx, gfx::Rectangle(x+1, top + lastl, cx+this_width-1, he - lastl));
        can.drawBar(gfx::Rectangle(bx + cx, top + lastl, 2,  he - lastl), blackColor, blackColor, gfx::FillPattern::SOLID, gfx::OPAQUE_ALPHA);

        ctx.setColor(state != Normal ? util::SkinColor::Heading : util::SkinColor::Static);
        outText(ctx, gfx::Point(x + 10, top + 1), tab.name);
    }
}

void
ui::widgets::TabBar::drawBlank(gfx::Canvas& can, gfx::Rectangle area)
{
    int x     = area.getLeftX();
    int right = area.getRightX();
    int bot   = area.getBottomY();

    gfx::ColorQuad_t whiteColor = m_root.colorScheme().getColor(ui::Color_White);
    gfx::ColorQuad_t grayColor  = m_root.colorScheme().getColor(ui::Color_Gray);
    can.drawHLine(gfx::Point(x, bot-2), right-x, whiteColor, gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
    can.drawHLine(gfx::Point(x, bot-1), right-x, grayColor,  gfx::SOLID_LINE, gfx::OPAQUE_ALPHA);
}
