/**
  *  \file test/server/talk/render/htmlrenderertest.cpp
  *  \brief Test for server::talk::render::HTMLRenderer
  */

#include "server/talk/render/htmlrenderer.hpp"

#include "afl/net/nullcommandhandler.hpp"
#include "afl/net/redis/hashkey.hpp"
#include "afl/net/redis/internaldatabase.hpp"
#include "afl/net/redis/stringfield.hpp"
#include "afl/net/redis/stringkey.hpp"
#include "afl/test/testrunner.hpp"
#include "server/talk/render/context.hpp"
#include "server/talk/render/options.hpp"

using server::talk::render::renderHTML;
using server::talk::TextNode;
using afl::net::redis::StringKey;
using afl::net::redis::HashKey;
using afl::net::redis::StringSetKey;

/** Test some code highlighting.
    This is bug #330 which applies to the highlighter, but we're testing the full stack here. */
AFL_TEST("server.talk.render.HTMLRenderer:code", a)
{
    afl::net::NullCommandHandler nch;
    server::talk::Root root(nch, server::talk::Configuration());
    server::talk::render::Context ctx(root, "u");
    server::talk::render::Options opts;

    root.keywordTable().add("ini.phost.GameName.link", "http://phost.de/phost4doc/config.html#GameName");
    root.keywordTable().add("ini.phost.GameName.info", "Name of the game");

    // forum:[code=pconfig.src]pHost.Gamename=foo
    TextNode n1(TextNode::maGroup, TextNode::miGroupRoot);
    n1.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParCode, "pconfig.src"));
    n1.children[0]->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "pHost.Gamename=foo"));
    a.checkEqual("01", server::talk::render::renderHTML(n1, ctx, opts, root),
                 "<pre><a href=\"http://phost.de/phost4doc/config.html#GameName\" title=\"Name of the game\" class=\"syn-name\">pHost.Gamename</a>=foo</pre>\n");

    // forum:[code=pconfig.src]%foo\nbar
    TextNode n2(TextNode::maGroup, TextNode::miGroupRoot);
    n2.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParCode, "pconfig.src"));
    n2.children[0]->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "%foo\nbar"));
    a.checkEqual("11", server::talk::render::renderHTML(n2, ctx, opts, root),
                 "<pre><span class=\"syn-sec\">%foo</span>\n<span class=\"syn-name\">bar</span></pre>\n");
}

/** Render plaintext. */
AFL_TEST("server.talk.render.HTMLRenderer:plaintext", a)
{
    // Environment
    afl::net::NullCommandHandler nch;
    server::talk::Root root(nch, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "u");   // context, required for quoting [not required?]
    const server::talk::render::Options opts;             // options [not required?]

    // A single paragraph containing just text
    TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
    TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
    TextNode& text(*par.children.pushBackNew(new TextNode(TextNode::maPlain, 0)));

    // Basic test
    {
        text.text = "hi mom";
        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>hi mom</p>\n");
    }

    // Looks like a tag
    {
        text.text = "a<b>c";
        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>a&lt;b&gt;c</p>\n");
    }

    // Ampersand
    {
        text.text = "a&c";
        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>a&amp;c</p>\n");
    }
}

/** Render some regular text. */
AFL_TEST("server.talk.render.HTMLRenderer:complex", a)
{
    // Environment
    afl::net::NullCommandHandler nch;
    server::talk::Root root(nch, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "u");   // context, required for quoting [not required?]
    const server::talk::render::Options opts;             // options [not required?]

    // Two paragraphs
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi"));

        TextNode& par2(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par2.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>hi</p>\n<p>mom</p>\n");
    }

    // Paragraph with inline formatting (bold)
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, TextNode::miInBold));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>hi <b>mom</b>!</p>\n");
    }

    // Same thing, italic
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, TextNode::miInItalic));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>hi <em>mom</em>!</p>\n");
    }

    // Same thing, strikethrough
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, TextNode::miInStrikeThrough));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("31", renderHTML(tn, ctx, opts, root), "<p>hi <s>mom</s>!</p>\n");
    }

    // Same thing, underlined
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, TextNode::miInUnderline));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("41", renderHTML(tn, ctx, opts, root), "<p>hi <u>mom</u>!</p>\n");
    }

    // Same thing, monospaced
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, TextNode::miInMonospace));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("51", renderHTML(tn, ctx, opts, root), "<p>hi <tt>mom</tt>!</p>\n");
    }

    // Same thing, invalid maInline
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInline, 99));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("61", renderHTML(tn, ctx, opts, root), "<p>hi mom!</p>\n");
    }

    // Same thing, colored
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIAColor, "#ff0000"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("71", renderHTML(tn, ctx, opts, root), "<p>hi <font color=\"#ff0000\">mom</font>!</p>\n");
    }

    // Same thing, font
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIAFont, "courier"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("81", renderHTML(tn, ctx, opts, root), "<p>hi <span style=\"font-family: courier;\">mom</span>!</p>\n");
    }

    // Same thing, font that needs quoting
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIAFont, "x&y"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("91", renderHTML(tn, ctx, opts, root), "<p>hi <span style=\"font-family: x&amp;y;\">mom</span>!</p>\n");
    }

    // Same thing, increased size
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIASize, "3"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("101", renderHTML(tn, ctx, opts, root), "<p>hi <span style=\"font-size: 195%;\">mom</span>!</p>\n");
    }

    // Same thing, reduced size
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIASize, "-1"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("111", renderHTML(tn, ctx, opts, root), "<p>hi <span style=\"font-size: 80%;\">mom</span>!</p>\n");
    }

    // Same thing, attributeless size
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, TextNode::miIASize, ""));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("121", renderHTML(tn, ctx, opts, root), "<p>hi mom!</p>\n");
    }

    // Same thing, invalid maInlineAttr
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "hi "));
        par.children.pushBackNew(new TextNode(TextNode::maInlineAttr, 99, "3"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "mom"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "!"));

        a.checkEqual("131", renderHTML(tn, ctx, opts, root), "<p>hi mom!</p>\n");
    }
}

/** Test rendering of links. */
AFL_TEST("server.talk.render.HTMLRenderer:link", a)
{
    // Environment
    afl::net::NullCommandHandler nch;
    server::talk::Root root(nch, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "u");   // context, required for quoting [not required?]
    const server::talk::render::Options opts;             // options [not required?]

    // A link with differing content and target
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUrl, "http://web"));
        par.children.back()->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>before <a href=\"http://web\" rel=\"nofollow\">text</a> after</p>\n");
    }

    // A link with no content (=shortened form)
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUrl, "http://web"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>before <a href=\"http://web\" rel=\"nofollow\">http://web</a> after</p>\n");
    }

    // Quoted link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUrl, "http://a/x<y>z"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>before <a href=\"http://a/x&lt;y&gt;z\" rel=\"nofollow\">http://a/x&lt;y&gt;z</a> after</p>\n");
    }

    // Invalid link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUrl, "javascript:alert(\"hello\")"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("31", renderHTML(tn, ctx, opts, root), "<p>before <span class=\"tfailedlink\">link javascript:alert(&quot;hello&quot;)</span> after</p>\n");
    }
}

/** Test specials. */
AFL_TEST("server.talk.render.HTMLRenderer:special", a)
{
    // Environment
    afl::net::NullCommandHandler nch;
    server::talk::Root root(nch, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "u");   // context, required for quoting [not required?]
    server::talk::render::Options opts;                   // options
    opts.setBaseUrl("http://base/path/");

    // Image link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maSpecial, TextNode::miSpecialImage, "http://xyz"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>before <img src=\"http://xyz\" /> after</p>\n");
    }

    // Bad image link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maSpecial, TextNode::miSpecialImage, "javascript:alert(\"hi\")"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("02", renderHTML(tn, ctx, opts, root), "<p>before <span class=\"tfailedlink\">image javascript:alert(&quot;hi&quot;)</span> after</p>\n");
    }

    // Image link with alt text
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maSpecial, TextNode::miSpecialImage, "http://xyz"))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "some text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("03", renderHTML(tn, ctx, opts, root), "<p>before <img src=\"http://xyz\" alt=\"some text\" /> after</p>\n");
    }

    // Break
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maSpecial, TextNode::miSpecialBreak));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>before <br /> after</p>\n");
    }

    // Smiley
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "before "));
        par.children.pushBackNew(new TextNode(TextNode::maSpecial, TextNode::miSpecialSmiley, "smile"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " after"));

        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>before <img src=\"http://base/path/res/smileys/smile.png\" width=\"16\" height=\"16\" alt=\":smile:\" /> after</p>\n");
    }
}

/** Test rendering user links. */
AFL_TEST("server.talk.render.HTMLRenderer:link:user", a)
{
    // Environment
    afl::net::redis::InternalDatabase db;
    server::talk::Root root(db, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "1000");
    server::talk::render::Options opts;
    opts.setBaseUrl("http://base/path/");

    // Create two users
    StringKey(db, "uid:fred").set("1000");
    StringKey(db, "uid:wilma").set("1001");
    StringKey(db, "user:1000:name").set("fred");
    StringKey(db, "user:1001:name").set("wilma");
    HashKey(db, "user:1000:profile").stringField("screenname").set("Fred F");
    HashKey(db, "user:1001:profile").stringField("screenname").set("Wilma F");

    // Regular user link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUser, "wilma"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>[ <a class=\"userlink\" href=\"http://base/path/userinfo.cgi/wilma\">Wilma F</a> ]</p>\n");
    }

    // Named user link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUser, "wilma"))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "Text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("02", renderHTML(tn, ctx, opts, root), "<p>[ <a class=\"userlink\" href=\"http://base/path/userinfo.cgi/wilma\">Text</a> ]</p>\n");
    }

    // Regular user link to user himself
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUser, "fred"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>[ <a class=\"userlink userlink-me\" href=\"http://base/path/userinfo.cgi/fred\">Fred F</a> ]</p>\n");
    }

    // Unknown user
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUser, "barney"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">user barney</span> ]</p>\n");
    }

    // Partial tree, just a paragraph fragment
    {
        TextNode tn(TextNode::maParagraph, TextNode::miParFragment);
        tn.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        tn.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkUser, "wilma"));
        tn.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("31", renderHTML(tn, ctx, opts, root), "[ <a class=\"userlink\" href=\"http://base/path/userinfo.cgi/wilma\">Wilma F</a> ]");
    }
}

/** Test more links. */
AFL_TEST("server.talk.render.HTMLRenderer:link:other", a)
{
    // Environment
    afl::net::redis::InternalDatabase db;
    server::talk::Root root(db, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "1000");
    server::talk::render::Options opts;
    opts.setBaseUrl("http://base/path/");

    // Create environment
    // - a game
    StringSetKey(db, "game:all").add("7");
    StringKey(db, "game:7:state").set("running");
    StringKey(db, "game:7:type").set("public");
    StringKey(db, "game:7:name").set("Seven of Nine");

    // - a forum
    StringSetKey(db, "forum:all").add("3");
    HashKey(db, "forum:3:header").stringField("name").set("Chat Room");

    // - a thread
    HashKey(db, "thread:9:header").stringField("subject").set("Hi There");
    HashKey(db, "thread:9:header").stringField("forum").set("3");

    // - a posting
    HashKey(db, "msg:12:header").stringField("subject").set("Re: Hi There");
    HashKey(db, "msg:12:header").stringField("thread").set("9");
    HashKey(db, "msg:13:header").stringField("subject").set("We can also use a very long title which will be abbreviated when linked");
    HashKey(db, "msg:13:header").stringField("thread").set("9");
    HashKey(db, "msg:14:header").stringField("subject").set("");
    HashKey(db, "msg:14:header").stringField("thread").set("9");

    // Forum link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkForum, "3"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/forum.cgi/3-Chat-Room\">Chat Room</a> ]</p>\n");
    }

    // Bad forum link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkForum, "4"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("02", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">forum 4</span> ]</p>\n");
    }

    // Named forum link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        TextNode& link(*par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkForum, "3")));
        link.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("11", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/forum.cgi/3-Chat-Room\">text</a> ]</p>\n");
    }

    // Bad named forum link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        TextNode& link(*par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkForum, "4")));
        link.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("12", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">text</span> ]</p>\n");
    }

    // Thread link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkThread, "9"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("21", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There\">Hi There</a> ]</p>\n");
    }

    // Bad thread link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkThread, "bad"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("22", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">thread bad</span> ]</p>\n");
    }

    // Named thread link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        TextNode& link(*par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkThread, "9")));
        link.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "label"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("31", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There\">label</a> ]</p>\n");
    }

    // Post link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkPost, "12"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("41", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p12\">Re: Hi There</a> ]</p>\n");
    }

    // Bad post link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkPost, "999"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("42", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">post 999</span> ]</p>\n");
    }

    // Post link that is abbreviated
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkPost, "13"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("43", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p13\">We can also use a very long...</a> ]</p>\n");
    }

    // Post without subject
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkPost, "14"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("43", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p14\">(no subject)</a> ]</p>\n");
    }

    // Named post link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        TextNode& link(*par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkPost, "12")));
        link.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("51", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p12\">text</a> ]</p>\n");
    }

    // Game link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkGame, "7"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("61", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/host/game.cgi/7-Seven-of-Nine\">Seven of Nine</a> ]</p>\n");
    }

    // Named game link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        TextNode& link(*par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkGame, "7")));
        link.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "play"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("71", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"http://base/path/host/game.cgi/7-Seven-of-Nine\">play</a> ]</p>\n");
    }

    // Bad game link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkGame, "17"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("81", renderHTML(tn, ctx, opts, root), "<p>[ <span class=\"tfailedlink\">game 17</span> ]</p>\n");
    }

    // Email link
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& par(*tn.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal)));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, "[ "));
        par.children.pushBackNew(new TextNode(TextNode::maLink, TextNode::miLinkEmail, "a@b.c"));
        par.children.pushBackNew(new TextNode(TextNode::maPlain, 0, " ]"));

        a.checkEqual("91", renderHTML(tn, ctx, opts, root), "<p>[ <a href=\"mailto:a@b.c\">a@b.c</a> ]</p>\n");
    }
}

/** Test quotes. */
AFL_TEST("server.talk.render.HTMLRenderer:quote", a)
{
    // Environment
    afl::net::redis::InternalDatabase db;
    server::talk::Root root(db, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "1000");
    server::talk::render::Options opts;
    opts.setBaseUrl("http://base/path/");

    // Create environment
    StringKey(db, "user:1000:name").set("fred");
    StringKey(db, "uid:fred").set("1000");

    // - a forum
    StringSetKey(db, "forum:all").add("3");
    HashKey(db, "forum:3:header").stringField("name").set("Chat Room");

    // - a thread
    HashKey(db, "thread:9:header").stringField("subject").set("Hi There");
    HashKey(db, "thread:9:header").stringField("forum").set("3");

    // - a posting
    HashKey(db, "msg:12:header").stringField("subject").set("Re: Hi There");
    HashKey(db, "msg:12:header").stringField("thread").set("9");

    // Existing user
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, "fred"))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root),
                     "<div class=\"attribution\"><a class=\"userlink userlink-me\" href=\"http://base/path/userinfo.cgi/fred\"></a>:</div>\n"
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }

    // Nonexisting user
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, "barney"))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("02", renderHTML(tn, ctx, opts, root),
                     "<div class=\"attribution\">barney:</div>\n"
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }

    // User and posting
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, "fred;12"))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("03", renderHTML(tn, ctx, opts, root),
                     "<div class=\"attribution\"><a class=\"userlink userlink-me\" href=\"http://base/path/userinfo.cgi/fred\"></a> in <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p12\">Re: Hi There</a>:</div>\n"
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }

    // Nonexistant user, existing posting
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, "barney;12"))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("04", renderHTML(tn, ctx, opts, root),
                     "<div class=\"attribution\">barney in <a href=\"http://base/path/talk/thread.cgi/9-Hi-There#p12\">Re: Hi There</a>:</div>\n"
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }

    // Existant user, nonexistant posting
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, "fred;77"))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("05", renderHTML(tn, ctx, opts, root),
                     "<div class=\"attribution\"><a class=\"userlink userlink-me\" href=\"http://base/path/userinfo.cgi/fred\"></a> in 77:</div>\n"
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }

    // No attribution
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupQuote, ""))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "text"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root),
                     "<blockquote><p>text</p>\n"
                     "</blockquote>\n");
    }
}


/** Test list. */
AFL_TEST("server.talk.render.HTMLRenderer:list", a)
{
    // Environment
    afl::net::redis::InternalDatabase db;
    server::talk::Root root(db, server::talk::Configuration());
    const server::talk::render::Context ctx(root, "1000");
    server::talk::render::Options opts;
    opts.setBaseUrl("http://base/path/");

    // Compact form
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& list(*tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupList)));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "first"));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "second"));

        a.checkEqual("01", renderHTML(tn, ctx, opts, root), "<ul><li>first</li>\n<li>second</li>\n</ul>");
    }

    // Compact form, with numbering
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& list(*tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupList, "1")));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "first"));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "second"));

        a.checkEqual("02", renderHTML(tn, ctx, opts, root), "<ol><li>first</li>\n<li>second</li>\n</ol>");
    }

    // Compact form, with custom numbering
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& list(*tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupList, "\"a\"")));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "first"));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "second"));

        a.checkEqual("03", renderHTML(tn, ctx, opts, root), "<ol type=\"&quot;a&quot;\"><li>first</li>\n<li>second</li>\n</ol>");
    }

    // Full form: an entry with multiple paragraphs causes the list to be rendered as li-containing-p.
    {
        TextNode tn(TextNode::maGroup, TextNode::miGroupRoot);
        TextNode& list(*tn.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupList)));
        TextNode& n1 = *list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem));
        n1.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "first top"));
        n1.children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "first bottom"));
        list.children.pushBackNew(new TextNode(TextNode::maGroup, TextNode::miGroupListItem))
            ->children.pushBackNew(new TextNode(TextNode::maParagraph, TextNode::miParNormal))
            ->children.pushBackNew(new TextNode(TextNode::maPlain, 0, "second"));

        a.checkEqual("04", renderHTML(tn, ctx, opts, root), "<ul><li><p>first top</p>\n<p>first bottom</p>\n</li>\n<li><p>second</p>\n</li>\n</ul>");
    }
}
