/**
  *  \file test/interpreter/vmio/objectloadertest.cpp
  *  \brief Test for interpreter::vmio::ObjectLoader
  */

#include "interpreter/vmio/objectloader.hpp"

#include "afl/charset/utf8charset.hpp"
#include "afl/except/fileproblemexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/nullfilesystem.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/sys/log.hpp"
#include "afl/test/testrunner.hpp"
#include "interpreter/arguments.hpp"
#include "interpreter/arraydata.hpp"
#include "interpreter/binaryoperation.hpp"
#include "interpreter/values.hpp"
#include "interpreter/vmio/nullloadcontext.hpp"
#include "interpreter/vmio/worldloadcontext.hpp"

using interpreter::BCORef_t;

/** Loading bytecode objects. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:BytecodeObject", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);

    // Test data
    static const uint8_t FILE[] = {
        /* Header: not parsed by ObjectLoader */
        /* 0x43, 0x43, 0x6f, 0x62, 0x6a, 0x1a, 0x64, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, */

        /* A byte code object with Id 1:
             Local i
             For i:=1 To 10 Do Print "hello" */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x06, 0x05, 0x00, 0x00, 0x00, 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x01, 0x49, 0xff, 0xff, 0x07,
        0x00, 0x00, 0x00, 0x01, 0x0a, 0x01, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0a, 0x00, 0x06,
        0x00, 0x00, 0x00, 0x10, 0x01, 0x0c, 0x00, 0x0e, 0x04, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x0d,
        0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x02, 0x03, 0x00, 0x07, 0x04, 0x74, 0x2e, 0x71,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00
    };
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify
    BCORef_t bco(testee.getBCO(1));

    a.checkEqual("01. getFileName",        bco->getFileName(), "t.q");
    a.checkEqual("02. getSubroutineName",  bco->getSubroutineName(), "");
    a.checkEqual("03. getMinArgs",         bco->getMinArgs(), 0U);
    a.checkEqual("04. getMaxArgs",         bco->getMaxArgs(), 0U);

    a.checkEqual("11. getLineNumber",      bco->getLineNumber(0), 1U);
    a.checkEqual("12. getLineNumber",      bco->getLineNumber(1), 1U);
    a.checkEqual("13. getLineNumber",      bco->getLineNumber(2), 2U);
    a.checkEqual("14. getLineNumber",      bco->getLineNumber(10), 2U);
    a.checkEqual("15. getLineNumber",      bco->getLineNumber(100), 2U);

    a.checkEqual("21. getNumInstructions", bco->getNumInstructions(), 12U);
    a.checkEqual("22. major",            (*bco)(0).major, interpreter::Opcode::maPush);
    a.checkEqual("23. minor",            (*bco)(0).minor, interpreter::Opcode::sBoolean);
    a.checkEqual("24. arg",              (*bco)(0).arg,   0xFFFFU);
    a.checkEqual("25. major",            (*bco)(5).major, interpreter::Opcode::maBinary);
    a.checkEqual("26. minor",            (*bco)(5).minor, interpreter::biCompareLE);
    a.checkEqual("27. arg",              (*bco)(5).arg,   0);

    a.checkEqual("31. getNumNames",        bco->names().getNumNames(), 1U);
    a.checkEqual("32. getName",           *bco->getNameByIndex(0), "I");

    a.checkEqual("41. literals",           bco->literals().size(), 1U);
    a.checkEqual("42. getLiteral",         interpreter::toString(bco->getLiteral(0), true), "\"hello\"");
}

/** Loading bytecode and hash object. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:Hash", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);

    // Test data
    static const uint8_t FILE[] = {
        /* VM header:
        0x43, 0x43, 0x76, 0x6d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
        0x3a, 0x31, 0x31, 0x3a, 0x31, 0x31, 0x1a, 0x64, 0x04, 0x00, 0x01, 0x00, 0x07, 0x00, */

        /* #2 BCO, 187 bytes (0xbb) */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xbb, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00, 0x06, 0x07, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00,
        0x00, 0x00, 0x78, 0x74, 0x65, 0x73, 0x74, 0x2e, 0x76, 0x6d, 0x79, 0x06, 0x53, 0x41, 0x56, 0x45,
        0x4d, 0x45, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x09, 0x05, 0x00, 0x00,
        0x0d, 0x0b, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x05, 0x02, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x09, 0x05, 0x00, 0x00, 0x0d, 0x0b, 0x01, 0x41,
        0x46, 0x4f, 0x4f, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,

        /* #1 BCO, 206 bytes (0xce) */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xce, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x0f, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00,
        0x00, 0x00, 0x78, 0x79, 0x03, 0x46, 0x4f, 0x4f, 0x01, 0x51, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x09, 0x0b, 0x00, 0x00, 0x15, 0x0b, 0x01, 0x00, 0x01, 0x0a, 0x01, 0x00, 0x05, 0x00, 0x01, 0x00,
        0x06, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x0b, 0x05, 0x02, 0x00, 0x05, 0x00, 0x02, 0x00,
        0x06, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x0b, 0x05, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x04, 0x05, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,

        /* #3 Hash, 40 bytes (0x28) */
        0x05, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x01, 0x78, 0x01, 0x79, 0x00, 0x02, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00,

        /* #0 Process (not relevant)
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xee, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x01, 0x00, 0x74, 0x65, 0x73, 0x74,
        0x73, 0x74, 0x6d, 0x74, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x03, 0x00, 0x00, 0x00, 0x01, 0x51,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x03, 0x00, 0x00, 0x00, 0x01, 0x41, 0x00, 0x8a, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x01, 0x00, 0x00, 0x00 */
    };
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify hash
    {
        afl::data::Hash::Ref_t h(testee.getHash(3));
        a.checkEqual("01. getNumNames", h->getKeys().getNumNames(), 2U);

        int32_t iv;
        a.check     ("11. int x",   interpreter::checkIntegerArg(iv, h->get("x")));
        a.checkEqual("12. value x", iv, 1);
        a.check     ("13. int y",   interpreter::checkIntegerArg(iv, h->get("y")));
        a.checkEqual("14. value y", iv, 2);
        a.checkNull ("15. get z",   h->get("z"));
        a.checkNull ("16. get X",   h->get("X"));
    }

    // Verify BCOs
    {
        BCORef_t bco(testee.getBCO(2));
        a.checkEqual("21. getSubroutineName",  bco->getSubroutineName(), "FOO");
        a.checkEqual("22. getNumInstructions", bco->getNumInstructions(), 11U);
    }
    {
        BCORef_t bco(testee.getBCO(1));
        a.checkEqual("23. getNumInstructions", bco->getNumInstructions(), 15U);
    }
}

/** Loading bytecode and an array object. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:Array", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);

    // Test data
    /* Created using
         sub foo(a)
            saveme "array.vm"
         endsub

         local rr(5,5)
         rr(3,3) := 1
         rr(2,2) := "a"
         foo rr
       in PCC2's teststmt. */
    static const uint8_t FILE[] = {
        /* VM header:
        0x43, 0x43, 0x76, 0x6d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
        0x3a, 0x31, 0x31, 0x3a, 0x31, 0x31, 0x1a, 0x64, 0x04, 0x00, 0x01, 0x00, 0x07, 0x00, */

        /* #2 BCO, 126 bytes (0x7e) */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x06, 0x08, 0x00, 0x00, 0x00, 0x61, 0x72, 0x72, 0x61, 0x79, 0x2e, 0x76, 0x6d, 0x06, 0x53,
        0x41, 0x56, 0x45, 0x4d, 0x45, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04,
        0x05, 0x01, 0x41, 0x46, 0x4f, 0x4f, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,

        /* #1 BCO, 216 bytes (0xd8) */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x13, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0x00, 0x61, 0x03, 0x46, 0x4f,
        0x4f, 0x02, 0x52, 0x52, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x09, 0x0b, 0x05, 0x00, 0x06, 0x00,
        0x05, 0x00, 0x06, 0x00, 0x02, 0x00, 0x13, 0x0b, 0x01, 0x00, 0x01, 0x0a, 0x03, 0x00, 0x06, 0x00,
        0x03, 0x00, 0x06, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x0b, 0x05,
        0x02, 0x00, 0x06, 0x00, 0x02, 0x00, 0x06, 0x00, 0x01, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x0b, 0x05, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x05,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,

        /* #3 Array, 147 bytes (0x93) */
        0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x93, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x13, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x01, 0x00,
        0x00, 0x00, 0x61,

        /* #0 Process (not relevant
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xef, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x99, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x01, 0x00, 0x74, 0x65, 0x73, 0x74,
        0x73, 0x74, 0x6d, 0x74, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x03, 0x00, 0x00, 0x00, 0x02, 0x52,
        0x52, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x03, 0x00, 0x00, 0x00, 0x01, 0x41, 0x00, 0x8a, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x01, 0x00, 0x00, 0x00 */
    };
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify hash
    {
        afl::base::Ref<interpreter::ArrayData> arr(testee.getArray(3));
        a.checkEqual("01. getNumDimensions", arr->getNumDimensions(), 2U);
        a.checkEqual("02. getDimension",     arr->getDimension(0), 5U);
        a.checkEqual("03. getDimension",     arr->getDimension(1), 5U);

        int32_t iv;
        String_t sv;
        a.check     ("11. integer",   interpreter::checkIntegerArg(iv, arr->content()[5*3+3]));
        a.checkEqual("12. int value", iv, 1);
        a.check     ("13. string",    interpreter::checkStringArg(sv, arr->content()[5*2+2]));
        a.checkEqual("14. str value", sv, "a");

        a.checkEqual("21. content",   arr->content().size(), size_t(5*3+3+1));
    }

    // Verify BCOs
    {
        BCORef_t bco(testee.getBCO(2));
        a.checkEqual("31. getSubroutineName", bco->getSubroutineName(), "FOO");
        a.checkEqual("32. getNumInstructions", bco->getNumInstructions(), 3U);
    }
    {
        BCORef_t bco(testee.getBCO(1));
        a.checkEqual("33. getNumInstructions", bco->getNumInstructions(), 19U);
    }
}

/* Test loading a structure type. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:StructureType", a)
{
    /* Created using "c2script -c" with this script:
         struct s
           q, r, s
         endstruct */
    static const uint8_t FILE[] = {
        /* Object file header:
        0x43, 0x43, 0x6f, 0x62, 0x6a, 0x1a, 0x64, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, */

        /* #3 Type, 22 bytes */
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x01, 0x51, 0x01, 0x52, 0x01, 0x53,

        /* #2 BCO (constructor), 118 bytes */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x0d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x16, 0x0b, 0x01, 0x00,
        0x01, 0x0b, 0x53, 0x73, 0x2e, 0x71, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00,

        /* #1 BCO (entry), 107 bytes */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x01, 0x53, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x09, 0x0b,
        0x73, 0x2e, 0x71, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify
    {
        afl::base::Ref<interpreter::StructureTypeData> d(testee.getStructureType(3));
        a.checkEqual("01. getNumNames", d->names().getNumNames(), 3U);
        a.checkEqual("02. name #0",     d->names().getNameByIndex(0), "Q");
        a.checkEqual("03. name #1",     d->names().getNameByIndex(1), "R");
        a.checkEqual("04. name #2",     d->names().getNameByIndex(2), "S");
    }

    // Verify BCOs
    {
        BCORef_t bco(testee.getBCO(2));
        a.checkEqual("11. getSubroutineName", bco->getSubroutineName(), "S");
        a.checkEqual("12. getNumInstructions", bco->getNumInstructions(), 3U);
    }
    {
        BCORef_t bco(testee.getBCO(1));
        a.checkEqual("13. getNumInstructions", bco->getNumInstructions(), 2U);
    }
}

/* Test loading a structure value. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:StructureValue", a)
{
    /* Created using this script:
         struct s
           q, r, s
         endstruct

         local a := s()
         a->r := 'hi'
         saveme "s.vm" */
    static const uint8_t FILE[] = {
        /* VM Header:
        0x43, 0x43, 0x76, 0x6d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
        0x3a, 0x31, 0x31, 0x3a, 0x31, 0x31, 0x1a, 0x64, 0x04, 0x00, 0x01, 0x00, 0x07, 0x00, */

        /* #3 Type, 22 bytes */
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x01, 0x51, 0x01, 0x52, 0x01, 0x53,

        /* #2 Constructor BCO, 0x73 bytes */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x0d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x16, 0x0b, 0x01, 0x00,
        0x01, 0x0b, 0x53, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00,

        /* #1 Entry point BCO, 0xC1 bytes */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04, 0x00,
        0x00, 0x00, 0x68, 0x69, 0x73, 0x2e, 0x76, 0x6d, 0x01, 0x53, 0x01, 0x41, 0x01, 0x52, 0x06, 0x53,
        0x41, 0x56, 0x45, 0x4d, 0x45, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x09, 0x0b, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x09, 0x05, 0x01, 0x00, 0x01, 0x0a, 0x01, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x03, 0x09, 0x02, 0x00, 0x05, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04,
        0x05, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
        0x00,

        /* #4 Structure Value. 0x2A bytes */
        0x06, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x68, 0x69,

        /* Process (not relevant)
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x01, 0x00, 0x74, 0x65, 0x73, 0x74,
        0x73, 0x74, 0x6d, 0x74, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x04, 0x00, 0x00, 0x00, 0x01, 0x41,
        0x00, 0x8a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0x00, 0x00, 0x00, 0x00 */
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify type
    afl::base::Ref<interpreter::StructureTypeData> d(testee.getStructureType(3));
    afl::base::Ref<interpreter::StructureValueData> v(testee.getStructureValue(4));

    a.checkEqual("01. getNumNames", d->names().getNumNames(), 3U);
    a.checkEqual("02. name #0",     d->names().getNameByIndex(0), "Q");
    a.checkEqual("03. name #1",     d->names().getNameByIndex(1), "R");
    a.checkEqual("04. name #2",     d->names().getNameByIndex(2), "S");

    a.checkEqual("11. type", &v->type(), &*d);
    a.checkEqual("12. data", interpreter::toString(v->data()[1], false), "hi");
}

/* Test loading a structure value, reversed order.
   This exercises forward references. */
AFL_TEST("interpreter.vmio.ObjectLoader:StructureValue:reversed", a)
{
    /* Modified from testLoadStructureValue */
    static const uint8_t FILE[] = {
        /* #4 Structure Value. 0x2A bytes */
        0x06, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x68, 0x69,

        /* #3 Type, 22 bytes */
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x01, 0x51, 0x01, 0x52, 0x01, 0x53,
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    testee.load(s);

    // Verify type
    afl::base::Ref<interpreter::StructureTypeData> d(testee.getStructureType(3));
    afl::base::Ref<interpreter::StructureValueData> v(testee.getStructureValue(4));

    a.checkEqual("01. getNumNames", d->names().getNumNames(), 3U);
    a.checkEqual("02. name #0",     d->names().getNameByIndex(0), "Q");
    a.checkEqual("03. name #1",     d->names().getNameByIndex(1), "R");
    a.checkEqual("04. name #2",     d->names().getNameByIndex(2), "S");

    a.checkEqual("11. type", &v->type(), &*d);
    a.checkEqual("12. data", interpreter::toString(v->data()[1], false), "hi");
}

/** Test loadObjectFile. */
AFL_TEST("interpreter.vmio.ObjectLoader:loadObjectFile", a)
{
    /* Same file as in testLoadStruct, this time with header. */
    static const uint8_t FILE[] = {
        /* Object file header: */
        0x43, 0x43, 0x6f, 0x62, 0x6a, 0x1a, 0x64, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00,

        /* #3 Type, 22 bytes */
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x01, 0x51, 0x01, 0x52, 0x01, 0x53,

        /* #2 BCO (constructor), 118 bytes */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x0d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x16, 0x0b, 0x01, 0x00,
        0x01, 0x0b, 0x53, 0x73, 0x2e, 0x71, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00,

        /* #1 BCO (entry), 107 bytes */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6b, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x01, 0x53, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x09, 0x0b,
        0x73, 0x2e, 0x71, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Testee
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    BCORef_t bco = testee.loadObjectFile(s);

    // Verify
    a.checkEqual("01. getNumInstructions", bco->getNumInstructions(), 2U);
}

/** Test loadObjectFile, error cases. */

// Empty file
AFL_TEST("interpreter.vmio.ObjectLoader:loadObjectFile:error:empty", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(afl::base::Nothing));
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    AFL_CHECK_THROWS(a, testee.loadObjectFile(s), afl::except::FileProblemException);
}

// Bad magic
AFL_TEST("interpreter.vmio.ObjectLoader:loadObjectFile:error:bad-magic", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    static const uint8_t FILE[] = {
        0x44, 0x44, 0x66, 0x66, 0x66, 0x11, 0x64, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00,
    };
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    AFL_CHECK_THROWS(a, testee.loadObjectFile(s), afl::except::FileProblemException);
}

// Truncated object
AFL_TEST("interpreter.vmio.ObjectLoader:loadObjectFile:error:truncated-object", a)
{
    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    static const uint8_t FILE[] = {
        0x43, 0x43, 0x6f, 0x62, 0x6a, 0x1a, 0x64, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    };
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
    AFL_CHECK_THROWS(a, testee.loadObjectFile(s), afl::except::FileProblemException);
}

/* Test loading a process. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:Process", a)
{
    /* Almost the same as testLoadStructureValue.
       The process contains a GlobalContext; our LoadContext used for loading does not have that. Removed it manually. */
    static const uint8_t FILE[] = {
        /* VM Header:
        0x43, 0x43, 0x76, 0x6d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x2d, 0x31, 0x31, 0x31, 0x31, 0x31, 0x31,
        0x3a, 0x31, 0x31, 0x3a, 0x31, 0x31, 0x1a, 0x64, 0x04, 0x00, 0x01, 0x00, 0x07, 0x00, */

        /* #3 Type, 22 bytes */
        0x07, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x01, 0x51, 0x01, 0x52, 0x01, 0x53,

        /* #2 Constructor BCO, 0x73 bytes */
        0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x73, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x0d, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x16, 0x0b, 0x01, 0x00,
        0x01, 0x0b, 0x53, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03,
        0x00, 0x00, 0x00,

        /* #1 Entry point BCO, 0xC1 bytes */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x07, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x06, 0x04, 0x00,
        0x00, 0x00, 0x68, 0x69, 0x73, 0x2e, 0x76, 0x6d, 0x01, 0x53, 0x01, 0x41, 0x01, 0x52, 0x06, 0x53,
        0x41, 0x56, 0x45, 0x4d, 0x45, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x09, 0x0b, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x09, 0x05, 0x01, 0x00, 0x01, 0x0a, 0x01, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x03, 0x09, 0x02, 0x00, 0x05, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04,
        0x05, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
        0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
        0x00,

        /* #4 Structure Value. 0x2A bytes */
        0x06, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x00, 0x68, 0x69,

        /* Process */
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9c-6,0x00,0x00, 0x00, 0x07, 0x00, 0x00, 0x00,

        /* - Property Headers */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
        0x02-1,0x00,0x00, 0x00, 0x0c-6,0x00,0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

        /* - Main Header, 4 bytes */
        0x32, 0x00, 0x01, 0x00,

        /* - Name, 8 bytes */
        0x74, 0x65, 0x73, 0x74, 0x73, 0x74, 0x6d, 0x74,

        /* - Frames, 0x4C bytes */
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x04, 0x00, 0x00, 0x00, 0x01, 0x41,

        /* - Contexts, 6 bytes (manually removed the GlobalContext) */
        /* 0x00, 0x8a, 0x00, 0x00, 0x00, 0x00, */
        0x00, 0x89, 0x00, 0x00, 0x00, 0x00
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;

    // Success case
    {
        afl::io::NullFileSystem fs;
        afl::sys::Log log;
        interpreter::World world(log, tx, fs);
        interpreter::ProcessList processList;
        interpreter::vmio::WorldLoadContext worldContext(ctx, processList, world);

        interpreter::vmio::ObjectLoader testee(cs, tx, worldContext);
        afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
        AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(s));

        a.checkEqual("11. getProcessList", processList.getProcessList().size(), 1U);
    }

    // Failure case: file is not accepted when there is no WorldLoadContext
    {
        interpreter::vmio::ObjectLoader testee(cs, tx, ctx);
        afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
        AFL_CHECK_THROWS(a("21. load"), testee.load(s), afl::except::FileProblemException);
    }
}

/** Test loading a process with a mutex. */
AFL_TEST("interpreter.vmio.ObjectLoader:load:Mutex", a)
{
    static const uint8_t FILE[] = {
        /* VM header:
        0x43, 0x43, 0x76, 0x6d, 0x30, 0x31, 0x2d, 0x30, 0x34, 0x2d, 0x32, 0x30, 0x32, 0x33, 0x32, 0x30,
        0x3a, 0x34, 0x31, 0x3a, 0x33, 0x30, 0x1a, 0x64, 0x04, 0x00, 0x0b, 0x00, 0x01, 0x00, */

        /* #1 BCO, 0x7c bytes */
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x06, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x06, 0x01, 0x00, 0x00, 0x00, 0x58, 0x04, 0x4c, 0x4f, 0x43, 0x4b, 0x00, 0x00, 0x05, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x09, 0x05, 0x00, 0x00, 0x02, 0x0b, 0x00, 0x00, 0x12, 0x0b,
        0x00, 0x00, 0x03, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,

        /* #0 Process, 0xba bytes */
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xba-6,0x00,0x00, 0x00, 0x07, 0x00, 0x00, 0x00,

        /* - Property headers */
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00,
        0x03-1,0x00,0x00, 0x00, 0x1b-6,0x00,0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

        /* - Main header, 4 bytes */
        0x32, 0x00, 0x01, 0x00,

        /* - Name, 0x1f bytes */
        0x43, 0x6f, 0x6e, 0x73, 0x6f, 0x6c, 0x65, 0x3a, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6c, 0x6f,
        0x63, 0x6b, 0x28, 0x22, 0x58, 0x22, 0x29, 0x20, 0x64, 0x6f, 0x20, 0x73, 0x74, 0x6f, 0x70,

        /* - Frames, 0x44 bytes */
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,

        /* - Contexts, 0x1b bytes (manually removed the GlobalContext) */
        /* 0x00, 0x8a, 0x00, 0x00, 0x00, 0x00, */
        0x00, 0x89, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x8b, 0x01, 0x00, 0x00, 0x00,

        /* - Context aux data: mutex named 'X' */
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x58
    };

    interpreter::vmio::NullLoadContext ctx;
    afl::charset::Utf8Charset cs;
    afl::string::NullTranslator tx;
    afl::io::NullFileSystem fs;
    afl::sys::Log log;
    interpreter::World world(log, tx, fs);
    interpreter::ProcessList processList;
    interpreter::vmio::WorldLoadContext worldContext(ctx, processList, world);

    interpreter::vmio::ObjectLoader testee(cs, tx, worldContext);
    afl::base::Ref<afl::io::Stream> s(*new afl::io::ConstMemoryStream(FILE));
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(s));

    // Must have produced a process
    a.checkEqual("11. getProcessList", processList.getProcessList().size(), 1U);

    // Must have produces a mutex 'X' owned by the process
    interpreter::MutexList::Mutex* mtx = world.mutexList().query("X");
    a.checkNonNull("21. mutex", mtx);
    a.checkEqual("22. getOwner", mtx->getOwner(), processList.getProcessList()[0]);
}
