/**
  *  \file test/gfx/gen/spaceviewconfigtest.cpp
  *  \brief Test for gfx::gen::SpaceViewConfig
  */

#include "gfx/gen/spaceviewconfig.hpp"

#include "afl/test/testrunner.hpp"
#include "util/randomnumbergenerator.hpp"
#include <cstdio>

namespace {
    void verify(afl::test::Assert a, const gfx::RGBAPixmap& pix, afl::base::Memory<const uint32_t> expect)
    {
        afl::base::Memory<const uint32_t> got(pix.pixels());
#if 1
        a.checkEqualContent("content", got, expect);
#else
        // Generate the test code
        std::printf("static const uint32_t %s[] = {\n", "<name>");
        while (!got.empty()) {
            afl::base::Memory<const uint32_t> row = got.split(16);
            while (const uint32_t* p = row.eat()) {
                std::printf(" 0x%08X,", *p);
            }
            std::printf("\n");
        }
        std::printf("};\n");
#endif
    }
}

/** Regression test. */
AFL_TEST("gfx.gen.SpaceViewConfig:normal", a)
{
    // Configure
    gfx::gen::SpaceViewConfig testee;
    testee.setSize(gfx::Point(32, 48));
    testee.setNumSuns(0);
    testee.setStarProbability(90);

    // Render
    util::RandomNumberGenerator rng(42);
    afl::base::Ref<gfx::RGBAPixmap> pix = testee.render(rng);

    // Verify
    static const uint32_t EXPECT[] = {
        0x4D334AFF, 0x472E44FF, 0x3D2739FF, 0x33202FFF, 0x2A1A26FF, 0x9C9098FF, 0x1F131BFF, 0x1C1119FF, 0x180F15FF, 0x160E13FF, 0x150E12FF, 0x150E12FF, 0x171014FF, 0x1A1417FF, 0x181013FF, 0x120C0EFF,
        0x0E090CFF, 0x0D080BFF, 0x0C080AFF, 0x0B0608FF, 0x0A0608FF, 0x0A0608FF, 0x0B0709FF, 0x413E40FF, 0x0D070AFF, 0x0E090BFF, 0x10090DFF, 0x130A0EFF, 0x140B0FFF, 0x130B0FFF, 0x0E080BFF, 0x080405FF,
        0x543850FF, 0x4C324AFF, 0x442D42FF, 0x3E283BFF, 0x382536FF, 0x2F1E2DFF, 0x281926FF, 0x231620FF, 0x1F141CFF, 0x1B1017FF, 0x191015FF, 0x191015FF, 0x1D1519FF, 0xF4EBF0FF, 0x1C1418FF, 0x120D10FF,
        0x0F0A0EFF, 0x0E090CFF, 0x0D080AFF, 0x0C0608FF, 0x0B0608FF, 0x0B0608FF, 0x0B0608FF, 0x0B0608FF, 0x0C0709FF, 0x0E080BFF, 0x0F090CFF, 0x0F090DFF, 0x100A0DFF, 0x0E090DFF, 0x0A0709FF, 0x060404FF,
        0x4A2F44FF, 0x563954FF, 0x4E344CFF, 0x442D42FF, 0x3D283BFF, 0x3A2637FF, 0x322030FF, 0x2B1C29FF, 0x291A25FF, 0x1F141AFF, 0x1D1219FF, 0x191015FF, 0x1B1216FF, 0x1E1519FF, 0x191114FF, 0x120C10FF,
        0x120C0EFF, 0x100B0DFF, 0x0E080AFF, 0x0D0709FF, 0x0D0709FF, 0x0B0607FF, 0x0B0607FF, 0x0C0608FF, 0x0C0709FF, 0x0C0709FF, 0x0D080AFF, 0x0E080BFF, 0x0F090CFF, 0x0C080CFF, 0x080508FF, 0x040304FF,
        0x624466FF, 0x5A3D59FF, 0x422B3EFF, 0x4B324AFF, 0x3E2839FF, 0xE3CFDDFF, 0x382535FF, 0x321F2EFF, 0x2E1D29FF, 0x231620FF, 0x1C1118FF, 0x191015FF, 0x181014FF, 0x191015FF, 0x170E12FF, 0x140D12FF,
        0x170D10FF, 0x130A0CFF, 0x0F0708FF, 0x0E0708FF, 0x0D0608FF, 0x0A0506FF, 0x0A0506FF, 0x0A0506FF, 0x090506FF, 0x0A0607FF, 0x0D0709FF, 0x0D0709FF, 0x0C0608FF, 0x090508FF, 0x060507FF, 0x040405FF,
        0x5C405FFF, 0x614364FF, 0x2D1B24FF, 0x3F293BFF, 0x321F2BFF, 0x362332FF, 0x34212EFF, 0x311F2CFF, 0x2F1E2CFF, 0x2B1C29FF, 0x21151CFF, 0x1B1016FF, 0x1A1016FF, 0x181015FF, 0x181015FF, 0x180F15FF,
        0x180E11FF, 0x10090CFF, 0x0B0607FF, 0x090505FF, 0x0B0607FF, 0x080405FF, 0x070303FF, 0x070303FF, 0x070304FF, 0x070404FF, 0x090506FF, 0x0A0607FF, 0x090505FF, 0x080506FF, 0x050405FF, 0x060405FF,
        0x4B3047FF, 0x5C3F5DFF, 0x2D1C24FF, 0x29181EFF, 0x27171DFF, 0x28181FFF, 0x2A1A23FF, 0x2F1C27FF, 0x311F2EFF, 0x311F2EFF, 0x2C1C29FF, 0x21141AFF, 0x1E1319FF, 0x1D1218FF, 0x1D1218FF, 0x1D1117FF,
        0x1A0F12FF, 0x0C070AFF, 0x090506FF, 0x060404FF, 0x080405FF, 0x070303FF, 0x050302FF, 0x040202FF, 0x040202FF, 0x060403FF, 0x070404FF, 0x070504FF, 0x0A0807FF, 0x0A0909FF, 0x080708FF, 0x060405FF,
        0x3B2129FF, 0x2E191EFF, 0x2E1D25FF, 0xD9BDD9FF, 0x29191FFF, 0x25151AFF, 0x32202CFF, 0x32202DFF, 0x362332FF, 0x372434FF, 0x32202FFF, 0x2A1924FF, 0x23151DFF, 0x22151CFF, 0x22141AFF, 0x221319FF,
        0x180E13FF, 0x0B080BFF, 0x070406FF, 0x050203FF, 0x050203FF, 0x050302FF, 0x030202FF, 0x020101FF, 0x020101FF, 0x040303FF, 0x050303FF, 0x070505FF, 0x0A0909FF, 0xFEFDFDFF, 0x0A0A0BFF, 0x070607FF,
        0x3F2530FF, 0x301A1FFF, 0x3C2532FF, 0x6B4B6EFF, 0x604362FF, 0x301D27FF, 0x36222EFF, 0xE0CADBFF, 0x452D42FF, 0x3D283AFF, 0x392535FF, 0x321F2EFF, 0x2A1924FF, 0x271721FF, 0x26171EFF, 0x25161CFF,
        0x181018FF, 0x0B080DFF, 0x060406FF, 0x060303FF, 0x040202FF, 0x030101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030202FF, 0x030202FF, 0x040303FF, 0x080707FF, 0x0A090AFF, 0x080809FF, 0x080609FF,
        0x412735FF, 0x301B23FF, 0x321D27FF, 0x5D3F5DFF, 0x68486CFF, 0x5B405DFF, 0x452E44FF, 0x4D354DFF, 0x4D334BFF, 0x422B40FF, 0x3E283BFF, 0x382434FF, 0x311F2DFF, 0x2D1B27FF, 0x2B1A25FF, 0x231620FF,
        0x18101AFF, 0x0C080DFF, 0x080508FF, 0x050304FF, 0x040202FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030202FF, 0x030303FF, 0x040404FF, 0x070506FF, 0x070608FF, 0x0A080CFF,
        0x442C40FF, 0x301D28FF, 0x27171EFF, 0x5C405FFF, 0x6E4E76FF, 0x694970FF, 0x624669FF, 0x5B4060FF, 0x543A56FF, 0x493048FF, 0x442D43FF, 0x3E283BFF, 0x362232FF, 0x33202FFF, 0x2C1B26FF, 0x1E141FFF,
        0x150F19FF, 0x0E0A10FF, 0x0A070BFF, 0x050305FF, 0x030102FF, 0x010000FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030202FF, 0x030202FF, 0x040303FF, 0x040405FF, 0x060507FF, 0x070609FF,
        0x49314BFF, 0x3F283BFF, 0x362333FF, 0x2A1A27FF, 0x70507BFF, 0x6B4C75FF, 0x65486FFF, 0x604468FF, 0x5A3F5FFF, 0x533A57FF, 0x4C344EFF, 0x442D44FF, 0x3D293DFF, 0x352333FF, 0x2A1C2BFF, 0x1C1420FF,
        0x161019FF, 0x100C13FF, 0x0C080DFF, 0x080508FF, 0x050204FF, 0x030201FF, 0x030101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x040303FF, 0x050304FF, 0x070507FF, 0x070407FF,
        0x5D446AFF, 0x624870FF, 0x62466AFF, 0x49344FFF, 0x654970FF, 0x3E2A40FF, 0x3D293EFF, 0x422C44FF, 0x523A58FF, 0x563D5DFF, 0x4C3551FF, 0x402B42FF, 0x3A2940FF, 0x302031FF, 0x281B2AFF, 0x1C1421FF,
        0x161019FF, 0x110C14FF, 0x0D090EFF, 0x0A060AFF, 0x060304FF, 0x040201FF, 0x020000FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030202FF, 0x040304FF, 0x090506FF,
        0x5E4872FF, 0x654D7AFF, 0x6C5180FF, 0x6F507DFF, 0x442F47FF, 0x261824FF, 0x20141CFF, 0x20151FFF, 0x271A27FF, 0x4E3856FF, 0x4C3653FF, 0x36263AFF, 0x2C1E2DFF, 0x271B29FF, 0x201725FF, 0x1A141FFF,
        0x130E16FF, 0x0C080DFF, 0x0A060AFF, 0x0A060AFF, 0x040204FF, 0x030101FF, 0x030100FF, 0x020000FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030101FF, 0x030201FF, 0x040201FF, 0x060303FF, 0x0E0707FF,
        0x564168FF, 0x5E4771FF, 0x664D78FF, 0x664C78FF, 0x432F49FF, 0x1E1319FF, 0x150D13FF, 0x140D12FF, 0x312236FF, 0x412F49FF, 0x312335FF, 0x201621FF, 0x251B29FF, 0x1F1622FF, 0x18121BFF, 0x130D15FF,
        0x0B080DFF, 0x080407FF, 0x070405FF, 0x090507FF, 0x070405FF, 0x020101FF, 0x0A0302FF, 0x050201FF, 0x020000FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x030201FF, 0x040201FF, 0x080403FF, 0x0E0707FF,
        0x4C395BFF, 0x574269FF, 0x614A75FF, 0x5F4872FF, 0x3A2C44FF, 0x181119FF, 0x110C10FF, 0x0C080AFF, 0x170F17FF, 0x36273EFF, 0x35263BFF, 0x2D2032FF, 0x2F2236FF, 0x231A26FF, 0x161018FF, 0x100B11FF,
        0x0A070AFF, 0x050304FF, 0x040202FF, 0x030201FF, 0x060404FF, 0x090403FF, 0x0C0503FF, 0x0B0402FF, 0x040101FF, 0x010000FF, 0x010000FF, 0x030101FF, 0x030201FF, 0x040201FF, 0x080403FF, 0x130806FF,
        0x453453FF, 0x513E61FF, 0x5A456CFF, 0x5E4970FF, 0x382B42FF, 0x211925FF, 0x110C10FF, 0x0B080AFF, 0x0C080AFF, 0x251A2AFF, 0x271C2DFF, 0x251B29FF, 0x241B28FF, 0xF6EEF8FF, 0x141014FF, 0x0D090EFF,
        0x0A070AFF, 0x050203FF, 0x030101FF, 0x030101FF, 0x030101FF, 0x020000FF, 0x040101FF, 0x050201FF, 0x020100FF, 0x020000FF, 0x010000FF, 0x010000FF, 0x020000FF, 0x030100FF, 0x0A0402FF, 0x140805FF,
        0x3E2F4AFF, 0x483756FF, 0x554265FF, 0xE2CCF3FF, 0x473752FF, 0x281E2CFF, 0x110D10FF, 0x0C090BFF, 0x0F0B0EFF, 0x100B0FFF, 0x181219FF, 0x100B0EFF, 0x191219FF, 0x161116FF, 0x0E0A0EFF, 0x0A0609FF,
        0x080507FF, 0x050203FF, 0x030201FF, 0x030101FF, 0x030201FF, 0x020100FF, 0x020100FF, 0x020000FF, 0x020000FF, 0x020000FF, 0x010000FF, 0x010000FF, 0x020000FF, 0x030100FF, 0x0D0503FF, 0x1A0A06FF,
        0x362942FF, 0x41324DFF, 0x4A3A59FF, 0x554265FF, 0x4D3C5BFF, 0x29202FFF, 0x181219FF, 0x0D090BFF, 0x0C090AFF, 0x0C080AFF, 0x0C080AFF, 0x0B0709FF, 0x0F0B0EFF, 0x0A0708FF, 0x090607FF, 0x070404FF,
        0x060304FF, 0x030201FF, 0x030201FF, 0x030201FF, 0x020100FF, 0x030100FF, 0x030100FF, 0x020100FF, 0x020000FF, 0x020000FF, 0x010000FF, 0x010000FF, 0x020000FF, 0x060302FF, 0x130705FF, 0x220D09FF,
        0x312439FF, 0x392B43FF, 0x42324DFF, 0x4D3B59FF, 0xE7D4F1FF, 0x322638FF, 0x191219FF, 0x0F0B0EFF, 0x0E0B0CFF, 0x0F0C0DFF, 0x0A0808FF, 0x090606FF, 0x080505FF, 0x070504FF, 0x060403FF, 0x060403FF,
        0x030201FF, 0x030201FF, 0x030201FF, 0x030101FF, 0x020100FF, 0x030100FF, 0x030100FF, 0x020100FF, 0x020000FF, 0x020000FF, 0x020000FF, 0x030201FF, 0x050201FF, 0x0E0504FF, 0x180A07FF, 0x2C110CFF,
        0x281F30FF, 0x32253AFF, 0x392B44FF, 0x41324DFF, 0x413149FF, 0x36273AFF, 0x2C1F30FF, 0x1B141CFF, 0x130F11FF, 0xFBF8F9FF, 0x0D0B0BFF, 0x090606FF, 0x070404FF, 0x060403FF, 0x050302FF, 0x050302FF,
        0x040201FF, 0x030201FF, 0x030201FF, 0x030201FF, 0x020000FF, 0x020100FF, 0x020100FF, 0x020000FF, 0x020000FF, 0x030100FF, 0x030100FF, 0x060201FF, 0x0A0404FF, 0x160907FF, 0x230E0AFF, 0x2F130CFF,
        0x251B29FF, 0x2A1E2FFF, 0x2C2134FF, 0x32263CFF, 0x2C2133FF, 0x3A2A41FF, 0x302334FF, 0x1E151EFF, 0x0E0A0CFF, 0x0E0B0CFF, 0x0C0809FF, 0x090607FF, 0x080505FF, 0x080404FF, 0x060303FF, 0x050302FF,
        0x050302FF, 0x040202FF, 0x030201FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020000FF, 0x020000FF, 0x030100FF, 0x030100FF, 0x020100FF, 0x060201FF, 0x0E0604FF, 0x1D0B08FF, 0x2F120CFF, 0x38160EFF,
        0x1E131BFF, 0x1E1520FF, 0x1B1420FF, 0x161018FF, 0x161019FF, 0x2D2032FF, 0x241925FF, 0x180F16FF, 0x090606FF, 0x0C080AFF, 0x0F0A0DFF, 0x0C080AFF, 0x0B0608FF, 0x080506FF, 0x070405FF, 0x060404FF,
        0x060303FF, 0x050302FF, 0x040201FF, 0x030100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x030100FF, 0x060201FF, 0x060201FF, 0x050202FF, 0x090402FF, 0x0C0503FF, 0x260F09FF, 0x38150EFF, 0x431A10FF,
        0x180D0FFF, 0x180C0FFF, 0x160D10FF, 0x100B0FFF, 0x100B11FF, 0x161018FF, 0x160F16FF, 0x100B0EFF, 0x0B0709FF, 0x130D13FF, 0x130D13FF, 0x130D13FF, 0x0F0B0FFF, 0x0D090DFF, 0x0C080BFF, 0x080506FF,
        0x060303FF, 0x050303FF, 0x040201FF, 0x030100FF, 0x030201FF, 0x030201FF, 0x040201FF, 0x050201FF, 0x0C0503FF, 0x060302FF, 0x080403FF, 0x0C0504FF, 0x0E0604FF, 0x2C110BFF, 0x3A160FFF, 0x421910FF,
        0x1B0C0CFF, 0x160A09FF, 0x1A0C0BFF, 0x120808FF, 0x0E0809FF, 0x0B070AFF, 0x0B080AFF, 0x080506FF, 0x110C10FF, 0x171019FF, 0x19121BFF, 0x171019FF, 0x171019FF, 0x140E15FF, 0x120C12FF, 0x0C080BFF,
        0x070404FF, 0x050303FF, 0x040201FF, 0x040201FF, 0x040201FF, 0x040201FF, 0x050202FF, 0x080404FF, 0x0D0604FF, 0x060303FF, 0x070303FF, 0x0B0504FF, 0x170A06FF, 0x2B110BFF, 0x39160EFF, 0x3E1810FF,
        0x230E0BFF, 0x200D0AFF, 0x1E0C09FF, 0x1D0C09FF, 0x170B0AFF, 0x0E0B0BFF, 0x090708FF, 0x0A0609FF, 0x0F0B10FF, 0x19121CFF, 0x1A131EFF, 0x1B131EFF, 0x1C141FFF, 0x1A121BFF, 0x170F16FF, 0x130D13FF,
        0x0F0A0EFF, 0x060203FF, 0x050202FF, 0x060303FF, 0x070405FF, 0x040101FF, 0x050202FF, 0x0A0505FF, 0x0D0606FF, 0x080505FF, 0x0F0706FF, 0x100706FF, 0x1C0C09FF, 0x28100BFF, 0x35140DFF, 0x39170FFF,
        0x250F0BFF, 0x230E0AFF, 0x220E0AFF, 0x1F0E0AFF, 0x1E100DFF, 0xF3E9E8FF, 0x0D0B0BFF, 0x0B070AFF, 0x0F0A0FFF, 0x19121CFF, 0x1D1522FF, 0x201725FF, 0x221825FF, 0x211723FF, 0x1B1219FF, 0x110A0FFF,
        0x0C0709FF, 0x070404FF, 0x070304FF, 0x060203FF, 0x050202FF, 0x050201FF, 0x080404FF, 0x0B0607FF, 0x0C0607FF, 0x0F0708FF, 0x190C0AFF, 0x1D0E0CFF, 0x27110CFF, 0x2B120DFF, 0x2E120DFF, 0x33140EFF,
        0x230E0AFF, 0x1B0B08FF, 0x190B07FF, 0x100806FF, 0x0E0808FF, 0x110B0BFF, 0x0D0909FF, 0x0B080BFF, 0x0E090EFF, 0x160F19FF, 0x1F1624FF, 0x221827FF, 0x271B29FF, 0x271A27FF, 0x1F141EFF, 0x160E13FF,
        0x0C0709FF, 0x080404FF, 0x070404FF, 0x080405FF, 0x070303FF, 0x070302FF, 0x0E0708FF, 0x10090AFF, 0x0E0706FF, 0x150A08FF, 0x1A0D0BFF, 0xE8D7D5FF, 0x23100DFF, 0x250F0CFF, 0x28110DFF, 0x2C120EFF,
        0x240E09FF, 0x200D09FF, 0x1E0C08FF, 0x180B07FF, 0x100806FF, 0x0F0807FF, 0x0D0809FF, 0x0D080BFF, 0x100A0EFF, 0x150F17FF, 0x1F1623FF, 0x231926FF, 0x281C2BFF, 0x261A28FF, 0x281B28FF, 0x1E141DFF,
        0x0F080CFF, 0x0A0506FF, 0x070404FF, 0x070303FF, 0x080404FF, 0x090303FF, 0x0B0605FF, 0x100708FF, 0x0E0604FF, 0x120806FF, 0x160B09FF, 0x1A0D0CFF, 0x1D0D0BFF, 0x220F0CFF, 0x240F0CFF, 0x26100EFF,
        0x27100AFF, 0x260F0AFF, 0x240E09FF, 0x210E09FF, 0x1C0C08FF, 0x150907FF, 0x130908FF, 0x140A0BFF, 0x140B0EFF, 0x1D1219FF, 0x221621FF, 0x271926FF, 0x2C1C28FF, 0x2E1E2CFF, 0x2D1E2DFF, 0x1D131BFF,
        0x120B0FFF, 0x0E080BFF, 0x070404FF, 0x070404FF, 0x070303FF, 0x080303FF, 0x080404FF, 0x0B0504FF, 0x0C0504FF, 0x0E0604FF, 0x110807FF, 0x130908FF, 0x190C0AFF, 0x1E0D0AFF, 0x210E0BFF, 0x26100DFF,
        0x230E09FF, 0x250F0AFF, 0x260F09FF, 0x260F09FF, 0x240E09FF, 0x220E09FF, 0x1C0C09FF, 0x1A0B0BFF, 0x1D0E0FFF, 0x221317FF, 0x25161CFF, 0x2A1A23FF, 0x2E1B27FF, 0x2F1E2BFF, 0x2F1E2DFF, 0x221621FF,
        0x1A121AFF, 0x120B0FFF, 0x090405FF, 0x070404FF, 0x070304FF, 0x070303FF, 0x080303FF, 0x0A0404FF, 0x0B0504FF, 0x0C0505FF, 0x0D0606FF, 0x0F0707FF, 0x140909FF, 0x1A0B0BFF, 0x1F0D0BFF, 0x230E0CFF,
        0x1A0B07FF, 0x1C0C08FF, 0x200D08FF, 0x230E08FF, 0x250E09FF, 0x260F0AFF, 0x250F0AFF, 0x230F0CFF, 0x231010FF, 0x251315FF, 0x251316FF, 0x27151AFF, 0x2B1921FF, 0x2E1C27FF, 0x2E1D2BFF, 0x271824FF,
        0x1D1119FF, 0x180E13FF, 0x0C0607FF, 0x080405FF, 0x070304FF, 0x070304FF, 0x070304FF, 0x070404FF, 0x090404FF, 0x0A0404FF, 0x0B0605FF, 0x0E0707FF, 0x120809FF, 0x170A0AFF, 0x1B0B0BFF, 0x1D0C0BFF,
        0x130805FF, 0x130806FF, 0x150906FF, 0x180906FF, 0x150805FF, 0x1F0D08FF, 0x230F0AFF, 0x26100BFF, 0x27110EFF, 0x281212FF, 0x271213FF, 0x251213FF, 0x261317FF, 0x28171DFF, 0x2C1A26FF, 0x2A1A26FF,
        0x23151EFF, 0x1B1015FF, 0x10080AFF, 0x0C0608FF, 0x0A0506FF, 0x070304FF, 0x060303FF, 0x070304FF, 0x080305FF, 0x090505FF, 0x0A0506FF, 0x0C0607FF, 0x100809FF, 0x140A0AFF, 0x170B0BFF, 0x1A0C0BFF,
        0x0D0503FF, 0x0D0604FF, 0x0E0604FF, 0x0E0604FF, 0x0E0604FF, 0x130806FF, 0xECDEDBFF, 0x1F0D0AFF, 0x230F0CFF, 0x281211FF, 0x281212FF, 0x261212FF, 0x2B171BFF, 0x2E1A23FF, 0x2D1B26FF, 0x26161EFF,
        0x22141AFF, 0x1B1014FF, 0x190D12FF, 0x130B0EFF, 0x0D070AFF, 0x0A0507FF, 0x0D090DFF, 0x080507FF, 0x080405FF, 0x080405FF, 0x0A0607FF, 0x0D0608FF, 0x10090AFF, 0x120A0BFF, 0x160B0CFF, 0x180C0CFF,
        0x100604FF, 0x0A0504FF, 0x080403FF, 0x080302FF, 0x0A0403FF, 0x0B0504FF, 0x0E0705FF, 0x120806FF, 0x180A08FF, 0x1E0E0DFF, 0x1F0E0FFF, 0x200F10FF, 0x2A161BFF, 0x321D27FF, 0x321E2AFF, 0x2F1C28FF,
        0x28171FFF, 0x211218FF, 0x1D1117FF, 0x180D12FF, 0x150C11FF, 0x150D14FF, 0x100B10FF, 0x0C080BFF, 0x090507FF, 0x090507FF, 0x0B0507FF, 0x0E080BFF, 0x10080BFF, 0x110A0DFF, 0x130A0DFF, 0x160B0DFF,
        0x170906FF, 0x0F0604FF, 0x070302FF, 0x040201FF, 0x050201FF, 0x060302FF, 0x070302FF, 0x0A0403FF, 0x0E0605FF, 0x0C0608FF, 0x11090CFF, 0x160B0DFF, 0x1F1014FF, 0x2C1923FF, 0x2D1B26FF, 0x2D1B27FF,
        0x2A1922FF, 0x23141AFF, 0x22141AFF, 0x271825FF, 0x1E131BFF, 0x1A1119FF, 0x150E15FF, 0x0C080CFF, 0x0B080BFF, 0x0D090DFF, 0x100B0FFF, 0x110B0FFF, 0x120B0EFF, 0x130C10FF, 0x160D10FF, 0x160C0EFF,
        0x1B0B07FF, 0x190A07FF, 0x0E0604FF, 0x050202FF, 0x030201FF, 0x030101FF, 0x030201FF, 0x030201FF, 0x040202FF, 0x050204FF, 0x090608FF, 0x0F090DFF, 0x160C10FF, 0x1E1117FF, 0x201218FF, 0x281822FF,
        0x2A1923FF, 0x26151AFF, 0x231319FF, 0x22141BFF, 0x1C1117FF, 0x1C1119FF, 0x1B121BFF, 0x140D14FF, 0x110C11FF, 0x1B131FFF, 0x1A121CFF, 0x181119FF, 0x191119FF, 0x180F16FF, 0x170F14FF, 0x170E12FF,
        0x1E0C08FF, 0x1E0C08FF, 0x160905FF, 0x040201FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x020101FF, 0x010101FF, 0x020102FF, 0x060406FF, 0x0A060AFF, 0x0F090DFF, 0x160E14FF, 0x1C121AFF, 0x23151DFF,
        0x231318FF, 0x211218FF, 0x231419FF, 0x23151EFF, 0x22151FFF, 0x1F141EFF, 0x1E141EFF, 0x1B131CFF, 0x18101AFF, 0x211826FF, 0x211826FF, 0x201824FF, 0x1C151FFF, 0x1A131BFF, 0x181018FF, 0x181015FF,
        0x1C0C08FF, 0x1E0C08FF, 0x150906FF, 0x040202FF, 0x020101FF, 0x010101FF, 0x010101FF, 0x010101FF, 0x010101FF, 0x020102FF, 0x030204FF, 0x040305FF, 0x080509FF, 0x0B080DFF, 0x0C0709FF, 0x12090CFF,
        0x180D10FF, 0x190F14FF, 0x211319FF, 0x23151DFF, 0x251822FF, 0x261926FF, 0x241826FF, 0x201723FF, 0x211726FF, 0x241A2AFF, 0x241B2BFF, 0x241C2BFF, 0x201824FF, 0x1D141FFF, 0x1A141CFF, 0x1B131AFF,
        0x120907FF, 0x180B07FF, 0x110806FF, 0x040202FF, 0x030201FF, 0x030201FF, 0x010101FF, 0x010101FF, 0x020202FF, 0x020102FF, 0x020203FF, 0x010101FF, 0x030204FF, 0x050406FF, 0x080508FF, 0x0D080CFF,
        0x110B0EFF, 0x130C12FF, 0x1B121BFF, 0x211622FF, 0x201624FF, 0x261927FF, 0x271B29FF, 0x281C2CFF, 0x271C2DFF, 0x261D2EFF, 0x292031FF, 0xF3EBFAFF, 0x251D2BFF, 0x221926FF, 0x201823FF, 0x201822FF,
        0x1D0F0CFF, 0x1B0D0AFF, 0x190B08FF, 0x110806FF, 0x0C0504FF, 0x050201FF, 0x010101FF, 0x010101FF, 0x030203FF, 0x010101FF, 0x000000FF, 0x010101FF, 0x020102FF, 0x020203FF, 0x040305FF, 0x070508FF,
        0x0C080DFF, 0x140E17FF, 0x1B1421FF, 0x241B2CFF, 0x261C2DFF, 0x2C1F32FF, 0x2E2034FF, 0x2A1F32FF, 0x271E31FF, 0x281E31FF, 0x292032FF, 0x292132FF, 0x261E2DFF, 0x241B2AFF, 0x241D2AFF, 0xF5EDF7FF,
        0xECDEDBFF, 0x20100DFF, 0x1F0E0AFF, 0x1A0B08FF, 0x100706FF, 0x040201FF, 0x030202FF, 0x040203FF, 0x020202FF, 0x010101FF, 0x010101FF, 0x020202FF, 0x010101FF, 0x020102FF, 0x030204FF, 0x08060AFF,
        0x0F0C13FF, 0x1E1726FF, 0x32263FFF, 0x352841FF, 0x32263FFF, 0x372A44FF, 0x342840FF, 0x31253DFF, 0x2D2338FF, 0x2B2135FF, 0x2A2134FF, 0x292032FF, 0x281E2FFF, 0x261D2DFF, 0x251D2BFF, 0x241D2AFF,
        0x23110DFF, 0x190D0BFF, 0x200D0AFF, 0x190B09FF, 0x080506FF, 0x070405FF, 0x070405FF, 0x060406FF, 0x030304FF, 0x020202FF, 0x020202FF, 0x020202FF, 0x030203FF, 0x020203FF, 0x030204FF, 0x0D0A10FF,
        0x201827FF, 0x32273FFF, 0x31253DFF, 0x433353FF, 0x483759FF, 0x443455FF, 0x3D2F4BFF, 0x382B46FF, 0x352841FF, 0x32263EFF, 0x2F243AFF, 0x2D2237FF, 0x2B2033FF, 0x291F31FF, 0x271E2EFF, 0x251C2CFF,
        0x200F0CFF, 0x210F0BFF, 0x1E0E0BFF, 0x1C0C0AFF, 0x070405FF, 0x080508FF, 0x0B0709FF, 0x0A060AFF, 0x060508FF, 0x030303FF, 0xFFFFFFFF, 0x040304FF, 0x030203FF, 0x020203FF, 0x040305FF, 0x0B080DFF,
        0x231B2BFF, 0x2D2338FF, 0x56426BFF, 0x4F3C62FF, 0x55416AFF, 0x544068FF, 0x4A385BFF, 0x413150FF, 0x3B2D4AFF, 0x382B46FF, 0x352841FF, 0x32273EFF, 0x31253CFF, 0x2F243AFF, 0x2F253AFF, 0x2D2236FF,
        0x26110EFF, 0x26100DFF, 0x1F0E0BFF, 0x13090AFF, 0x0B0507FF, 0x0C0709FF, 0x0E080CFF, 0x0C080BFF, 0x070508FF, 0x030304FF, 0x040405FF, 0x030203FF, 0x030304FF, 0x020203FF, 0x060407FF, 0x120E16FF,
        0x2F243BFF, 0x413251FF, 0x644D7DFF, 0x5E4874FF, 0x5F4976FF, 0x644C7CFF, 0x59446FFF, 0x4E3C61FF, 0x473657FF, 0x413251FF, 0x3D2F4BFF, 0x392C47FF, 0x3A2C48FF, 0x362A43FF, 0x2C2135FF, 0x231B2BFF,
        0x271210FF, 0x231010FF, 0x271110FF, 0x24100FFF, 0x180C0DFF, 0x160C0EFF, 0x10090CFF, 0x0B0709FF, 0x080508FF, 0x060407FF, 0x040305FF, 0x040305FF, 0x030204FF, 0x040305FF, 0x060407FF, 0x120E16FF,
        0x372A44FF, 0x523E65FF, 0x6F558AFF, 0x71568CFF, 0x6B5284FF, 0x6D5488FF, 0x674F80FF, 0x5D4773FF, 0x554169FF, 0x4E3C61FF, 0x483759FF, 0x443455FF, 0x423150FF, 0x3B2C48FF, 0x2B2033FF, 0x231B2BFF,
        0x261211FF, 0x241111FF, 0x231010FF, 0x24100DFF, 0x251111FF, 0x201012FF, 0x160D10FF, 0x120B0EFF, 0x0C070AFF, 0x060407FF, 0x050406FF, 0x040305FF, 0x040305FF, 0x060407FF, 0x08060AFF, 0x120E16FF,
        0x4C3A5EFF, 0x664E7EFF, 0x6E5488FF, 0x695082FF, 0x785C94FF, 0x71568CFF, 0x6C5286FF, 0x685081FF, 0x614A78FF, 0x5A4570FF, 0x534067FF, 0x4D3B5FFF, 0x483757FF, 0x3D2E4BFF, 0x2D2237FF, 0x251C2DFF,
        0x351613FF, 0x2E1513FF, 0x2B1414FF, 0x281211FF, 0x2A1413FF, 0x291416FF, 0x221216FF, 0x1B1014FF, 0x120B10FF, 0x0D080DFF, 0x0B080DFF, 0x09070CFF, 0x08060AFF, 0x09070BFF, 0x211928FF, 0x362943FF,
        0x59446EFF, 0x6A5284FF, 0x7A5E98FF, 0x7A5D96FF, 0x795D96FF, 0x71578CFF, 0x6C5386FF, 0x6C5286FF, 0x674F80FF, 0x614A79FF, 0x5B4570FF, 0x554169FF, 0x4E3C61FF, 0x443353FF, 0x31243BFF, 0x32243AFF,
        0x3D1914FF, 0x381815FF, 0x331715FF, 0x301616FF, 0x2F1617FF, 0x2E1719FF, 0x2C171BFF, 0x25151AFF, 0x1E1219FF, 0x160E15FF, 0x110D15FF, 0x110D16FF, 0x120E17FF, 0x17121CFF, 0x2F243BFF, 0x433353FF,
        0x5B4671FF, 0x70558AFF, 0x7C609AFF, 0xD6B8F4FF, 0x7A5E97FF, 0x6F5589FF, 0x71568CFF, 0x70558AFF, 0x6B5285FF, 0x634C7BFF, 0x5E4874FF, 0x57426CFF, 0x503D64FF, 0x453556FF, 0x34273FFF, 0x3C2D49FF,
    };
    verify(a, *pix, EXPECT);
}

/** Regression test (default config). */
AFL_TEST("gfx.gen.SpaceViewConfig:default", a)
{
    // Configure (size only)
    gfx::gen::SpaceViewConfig testee;
    testee.setSize(gfx::Point(32, 48));

    // Render
    util::RandomNumberGenerator rng(888);
    afl::base::Ref<gfx::RGBAPixmap> pix = testee.render(rng);

    // Verify
    static const uint32_t EXPECT[] = {
        0x0A0B08FF, 0x090B07FF, 0x090B07FF, 0x090A06FF, 0x080906FF, 0x090A06FF, 0x090A06FF, 0x090B07FF, 0x090B07FF, 0x0A0C08FF, 0x050603FF, 0x040402FF, 0x040402FF, 0x040402FF, 0x030201FF, 0x040302FF,
        0x040302FF, 0x040302FF, 0x040402FF, 0x050503FF, 0x070806FF, 0x0F120EFF, 0x161A15FF, 0x1A1E18FF, 0x151813FF, 0x151914FF, 0x1A1E18FF, 0x1D231CFF, 0x151713FF, 0x131612FF, 0x141612FF, 0x0C0D0BFF,
        0x0C0E09FF, 0x0A0C08FF, 0x0A0B08FF, 0x0A0A06FF, 0x0A0A06FF, 0x0A0A06FF, 0x0A0B07FF, 0x0A0A07FF, 0x080805FF, 0x070604FF, 0x050502FF, 0x050402FF, 0x040402FF, 0x040402FF, 0x040502FF, 0x050603FF,
        0x040402FF, 0x040302FF, 0x070705FF, 0xFFFFFDFF, 0x10110EFF, 0x11140FFF, 0x131712FF, 0x141712FF, 0x161A14FF, 0x161A15FF, 0x1D231CFF, 0x222820FF, 0x1E231CFF, 0x1C201BFF, 0x1C201BFF, 0x91948FFF,
        0x0F100BFF, 0x0D0E09FF, 0x0B0C08FF, 0x0A0B07FF, 0x0A0A06FF, 0x0A0B07FF, 0x0A0B07FF, 0x0A0A07FF, 0x080805FF, 0x060603FF, 0x050502FF, 0x050402FF, 0x050402FF, 0x050402FF, 0x050502FF, 0x050603FF,
        0x070805FF, 0x060704FF, 0x080806FF, 0x0C0D0AFF, 0x0E0F0CFF, 0x10110EFF, 0x11140FFF, 0x131611FF, 0x151813FF, 0x191D17FF, 0x1E231CFF, 0x232921FF, 0x282F26FF, 0x2B3229FF, 0x2A3128FF, 0x272D25FF,
        0x10110DFF, 0x0E0F0BFF, 0x0D0E09FF, 0x0B0C08FF, 0x0C0D09FF, 0x0A0C07FF, 0x0A0B07FF, 0x090A06FF, 0x080905FF, 0x070804FF, 0x060703FF, 0x050402FF, 0x050402FF, 0x050402FF, 0x050502FF, 0x060603FF,
        0x070704FF, 0x080906FF, 0x0A0C08FF, 0x0C0E0AFF, 0x0D0E0BFF, 0x0E0F0CFF, 0x0F110DFF, 0x11140FFF, 0x131712FF, 0x161A14FF, 0x1A1E18FF, 0x1D231CFF, 0x232921FF, 0x272E25FF, 0x2B3329FF, 0x2F372DFF,
        0x12140FFF, 0x10130DFF, 0x0F100BFF, 0x0D0D09FF, 0x0B0C07FF, 0x0B0B07FF, 0x0A0A06FF, 0x0A0A05FF, 0x080804FF, 0x080804FF, 0x070703FF, 0x060602FF, 0x060502FF, 0x050502FF, 0x050502FF, 0x050502FF,
        0x070604FF, 0x090905FF, 0x0B0C09FF, 0x0C0E0AFF, 0x0D0F0BFF, 0x0E0E0BFF, 0x0E0F0CFF, 0x10110EFF, 0x11140FFF, 0x131712FF, 0x161A14FF, 0x191D17FF, 0x1D221BFF, 0x222820FF, 0x282F26FF, 0x2D342BFF,
        0x13150FFF, 0x11130DFF, 0x0F100BFF, 0x0D0D09FF, 0x0B0C07FF, 0x0B0B05FF, 0x0A0A05FF, 0x090904FF, 0x090804FF, 0x080803FF, 0x060602FF, 0x060602FF, 0x060602FF, 0x060602FF, 0x060602FF, 0x050602FF,
        0x070604FF, 0x0A0906FF, 0x0B0C08FF, 0x0A0A07FF, 0x0A0B08FF, 0x0C0D0AFF, 0x0D0E0BFF, 0x0F110DFF, 0x10110EFF, 0x11140FFF, 0x131712FF, 0x171A15FF, 0x1A1F19FF, 0x1C211BFF, 0x21261FFF, 0x272E25FF,
        0x14160FFF, 0x12130EFF, 0x11120CFF, 0x0E0E09FF, 0x0C0C07FF, 0x0B0B05FF, 0x0B0A05FF, 0x0A0904FF, 0x0B0905FF, 0x090804FF, 0x080803FF, 0x080803FF, 0x080803FF, 0x070703FF, 0x070703FF, 0x070703FF,
        0x080704FF, 0x070604FF, 0x070704FF, 0x090806FF, 0x0B0B08FF, 0x0C0E0AFF, 0x0D0E0AFF, 0x0E0F0CFF, 0x0F110DFF, 0x10110EFF, 0x121410FF, 0x131612FF, 0x141813FF, 0x191E18FF, 0x1F241DFF, 0x222820FF,
        0x151710FF, 0x14160FFF, 0x13140DFF, 0x11120BFF, 0x0E0F08FF, 0x0D0D07FF, 0x0D0C06FF, 0x0E0C06FF, 0x0D0C06FF, 0x0B0A05FF, 0x0B0A05FF, 0x0A0904FF, 0x0A0804FF, 0x090804FF, 0x090804FF, 0x080704FF,
        0x090805FF, 0x080804FF, 0x080905FF, 0x0C0B09FF, 0x0F0E0BFF, 0x0E0E0BFF, 0x0D0E0BFF, 0x0E100CFF, 0x0E0F0CFF, 0x10110EFF, 0x12130FFF, 0x131612FF, 0x121511FF, 0x131611FF, 0x181B16FF, 0x1F241DFF,
        0x1A1B13FF, 0x181911FF, 0x181911FF, 0x14140DFF, 0x12130BFF, 0x111009FF, 0x12120AFF, 0x12120AFF, 0x101008FF, 0x100E07FF, 0x0D0D06FF, 0x0D0C06FF, 0x0C0C06FF, 0x0C0B06FF, 0x0A0804FF, 0x0A0905FF,
        0x0A0805FF, 0x090805FF, 0x090905FF, 0x0E0E0BFF, 0xFFFFFEFF, 0x10100DFF, 0x0C0E0AFF, 0x0E100CFF, 0x0F0F0CFF, 0x0F0F0CFF, 0x10110EFF, 0x131511FF, 0x171A16FF, 0x191C18FF, 0x171A15FF, 0x191D18FF,
        0x1C1F16FF, 0x1D1E15FF, 0xFFFFF6FF, 0x191911FF, 0x16160DFF, 0x15150CFF, 0x17170DFF, 0x16150BFF, 0x131209FF, 0x15140BFF, 0x15140CFF, 0x12110AFF, 0x111009FF, 0x0F0E08FF, 0x0C0C06FF, 0x0B0905FF,
        0x0B0A06FF, 0x0B0A06FF, 0x0A0906FF, 0x0E0D09FF, 0x0F0F0CFF, 0x0E0E0BFF, 0x0E0E0BFF, 0x0E0F0BFF, 0x0E100CFF, 0x0F100DFF, 0x10110EFF, 0x121310FF, 0x171A16FF, 0xF9FBF8FF, 0x1A1C18FF, 0x171B16FF,
        0x202419FF, 0x1E2015FF, 0x1F2015FF, 0x1C1C11FF, 0x1A190EFF, 0x1A190EFF, 0x1C1C10FF, 0x1B1B0FFF, 0x19180DFF, 0x19180DFF, 0x1C1A10FF, 0x1C1C12FF, 0x18170FFF, 0x13130BFF, 0x101109FF, 0x0E0D07FF,
        0x0D0C07FF, 0x0A0A05FF, 0x0C0B07FF, 0x0C0C08FF, 0x0C0C08FF, 0x0B0B08FF, 0x0C0C09FF, 0x0D0D0AFF, 0x0E100CFF, 0x10110EFF, 0x11120FFF, 0x11120FFF, 0x141613FF, 0x171A16FF, 0x161915FF, 0x161915FF,
        0x24271BFF, 0x222418FF, 0x212115FF, 0x1F2012FF, 0x1D1D10FF, 0x1F1E0FFF, 0x212112FF, 0x232414FF, 0x212011FF, 0x1F1E10FF, 0x222115FF, 0xFFFFF9FF, 0x201F15FF, 0x19190FFF, 0x17170EFF, 0x12130BFF,
        0x11100AFF, 0x0E0E08FF, 0x0D0D08FF, 0x0C0C08FF, 0x0C0C08FF, 0x0C0D09FF, 0x0C0C09FF, 0x0D0E0AFF, 0x0E0F0BFF, 0x121410FF, 0x151513FF, 0x121210FF, 0x11120FFF, 0x121411FF, 0x131511FF, 0x131612FF,
        0x282C1EFF, 0x262819FF, 0x252617FF, 0x232414FF, 0x232312FF, 0x252411FF, 0x282814FF, 0x282814FF, 0x292814FF, 0x292916FF, 0x292817FF, 0x282819FF, 0x252517FF, 0x222215FF, 0x1C1D12FF, 0x18180EFF,
        0x13140CFF, 0x0F0F09FF, 0x0E0E08FF, 0x0D0D08FF, 0x0C0C08FF, 0x0D0E09FF, 0x0C0D09FF, 0x0D0D0AFF, 0x10100CFF, 0x141612FF, 0xFDFDFBFF, 0x141411FF, 0x10100EFF, 0x0F110EFF, 0x10130FFF, 0x11130FFF,
        0x2E3121FF, 0x2F3220FF, 0x2C2E1BFF, 0x2A2A16FF, 0x2A2914FF, 0x2C2A12FF, 0x302E15FF, 0x323016FF, 0x333218FF, 0x34331AFF, 0x31311BFF, 0x2D2D19FF, 0x2A2B19FF, 0x292A1AFF, 0x242517FF, 0x1E1F13FF,
        0x181910FF, 0x11110AFF, 0x10110AFF, 0x0C0D07FF, 0x0D0E09FF, 0x0E0E0AFF, 0x0D0E0AFF, 0x0E0E0AFF, 0x0F0F0BFF, 0x11130FFF, 0x131511FF, 0x11110FFF, 0x0F0F0CFF, 0x0E100DFF, 0x0E100DFF, 0x0E100DFF,
        0x333725FF, 0x333723FF, 0x32341EFF, 0x313119FF, 0x343116FF, 0x383414FF, 0x3E3917FF, 0x423F1BFF, 0x433F1CFF, 0x403E1DFF, 0x3A381CFF, 0x35351CFF, 0xFFFFEFFF, 0x2F321FFF, 0x2A2C1CFF, 0x25271AFF,
        0x1F2115FF, 0x15160DFF, 0x11120BFF, 0x12110BFF, 0x11120CFF, 0x0F100AFF, 0x0F100BFF, 0x0D0E0AFF, 0x0E0F0BFF, 0x0E100CFF, 0x0F110DFF, 0x0D0D0BFF, 0x0C0C0AFF, 0x0C0D0AFF, 0x0B0D0AFF, 0x0A0C09FF,
        0x323622FF, 0x373A23FF, 0x38381FFF, 0x3B3A1BFF, 0x413D19FF, 0x494217FF, 0x534A17FF, 0x5A521CFF, 0x59511EFF, 0x534E21FF, 0x47441FFF, 0x414122FF, 0x3F4126FF, 0x383A24FF, 0x343724FF, 0x2F3322FF,
        0x25271AFF, 0x16170EFF, 0x14140DFF, 0x11120BFF, 0x0F100AFF, 0x10110CFF, 0x10110CFF, 0x0F0F0BFF, 0x0E0F0BFF, 0x0E0F0BFF, 0x0D0F0BFF, 0x0D0D0AFF, 0x0C0C0AFF, 0x0A0B08FF, 0x0A0B09FF, 0x080907FF,
        0x31331FFF, 0x383921FF, 0x403F21FF, 0x46441FFF, 0x534B1BFF, 0x665919FF, 0x7F6E1BFF, 0x8D791EFF, 0x857520FF, 0x706524FF, 0x5B5523FF, 0x4D4C26FF, 0x4A4B2BFF, 0x46492EFF, 0x3D412BFF, 0x353927FF,
        0x2C2E20FF, 0x1B1B11FF, 0x191A11FF, 0x16170FFF, 0x11120CFF, 0x12130EFF, 0x11130DFF, 0x10120DFF, 0x0F0F0BFF, 0x0E0F0BFF, 0x0C0D0AFF, 0x0C0C0AFF, 0x0A0A08FF, 0x0A0A08FF, 0x080A07FF, 0x070806FF,
        0x30311CFF, 0x38391EFF, 0x444220FF, 0x524D1FFF, 0x6A5F1EFF, 0x98821CFF, 0xE5BE1EFF, 0xFFEA23FF, 0xEDC725FF, 0xA38F26FF, 0x756B28FF, 0x5C5829FF, 0x53532EFF, 0x494D2FFF, 0x464B31FF, 0x343624FF,
        0x2D2F20FF, 0x1C1D13FF, 0xF9FCF0FF, 0x24271DFF, 0x181A12FF, 0x14150EFF, 0x12130EFF, 0x10130DFF, 0x0F100CFF, 0x0F0F0BFF, 0x0D0F0BFF, 0x0B0B09FF, 0x090907FF, 0x080806FF, 0x080907FF, 0x080907FF,
        0x2C2D18FF, 0x37361BFF, 0x45411EFF, 0x5B541FFF, 0x82721EFF, 0xE7BF1FFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xF0CB28FF, 0x8E7F29FF, 0x68632CFF, 0x575730FF, 0x525635FF, 0x3E4229FF, 0x2C2E1CFF,
        0x212215FF, 0x1D1C12FF, 0x1F2017FF, 0x282B20FF, 0x20231AFF, 0x191C14FF, 0x151711FF, 0x12150FFF, 0x10110CFF, 0x0F100CFF, 0x0D0F0AFF, 0x0B0C09FF, 0x0A0B08FF, 0x090907FF, 0x070806FF, 0x070806FF,
        0x292814FF, 0x343318FF, 0x43401BFF, 0x5B531CFF, 0x8C781EFF, 0xFFE51FFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFF129FF, 0x988629FF, 0x69642AFF, 0x585930FF, 0x4E5131FF, 0x3A3D25FF, 0x2A2B1AFF,
        0x202114FF, 0x1C1C11FF, 0x181910FF, 0x24281DFF, 0x23261CFF, 0x1C1F16FF, 0x171912FF, 0x13160FFF, 0x11130EFF, 0x10120DFF, 0x0F110CFF, 0x0D0F0AFF, 0x0D0E0AFF, 0x0B0C09FF, 0x080907FF, 0x070806FF,
        0x252511FF, 0x302E15FF, 0x3F3A18FF, 0x564E1AFF, 0x7B6816FF, 0xE2BA1BFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xEDC725FF, 0xFFFFECFF, 0x656029FF, 0x54532DFF, 0x4C4F30FF, 0x42462CFF, 0x313421FF,
        0x27281AFF, 0x25261AFF, 0x1D1F15FF, 0x26291EFF, 0x21231AFF, 0x1A1D14FF, 0x171912FF, 0x141711FF, 0x131510FF, 0x13150FFF, 0x10130EFF, 0x0F110DFF, 0x0F110DFF, 0x0E0F0CFF, 0x0B0D0AFF, 0x090B08FF,
        0x21200EFF, 0x2B2912FF, 0x363213FF, 0x453D13FF, 0x5F5213FF, 0x937B16FF, 0xE1B91AFF, 0xFFE41EFF, 0xE7BF1FFF, 0x9D8720FF, 0x706523FF, 0x595526FF, 0x4E4E2AFF, 0x4E5334FF, 0x484D33FF, 0x383B28FF,
        0x2F3222FF, 0x292C1FFF, 0x1D1F15FF, 0x22261BFF, 0x1B1E16FF, 0x171A12FF, 0x181A13FF, 0x181C14FF, 0x171A14FF, 0x161812FF, 0x141812FF, 0x161813FF, 0x141712FF, 0x12140FFF, 0x0F110EFF, 0x0D0F0BFF,
        0x1D1B0BFF, 0x25230EFF, 0x2F2B10FF, 0x37320FFF, 0x463D0FFF, 0x5E5012FF, 0x786614FF, 0x867118FF, 0x7E6C19FF, 0x695C1CFF, 0x585120FF, 0x4B4923FF, 0x474728FF, 0x494D31FF, 0x464C34FF, 0x373B29FF,
        0x303324FF, 0x282C1EFF, 0x22261BFF, 0x22261BFF, 0x1A1D15FF, 0x191C15FF, 0x1B1F17FF, 0x1C2119FF, 0x1A1E17FF, 0x1A1D16FF, 0x171B15FF, 0x1A1D17FF, 0x181B15FF, 0x151712FF, 0x111410FF, 0x0F120EFF,
        0x181709FF, 0x1F1D0BFF, 0x26230DFF, 0x2B280CFF, 0x352F0DFF, 0x40380EFF, 0x4C4210FF, 0x514813FF, 0x514815FF, 0x4B4418FF, 0x45421DFF, 0x414121FF, 0x414428FF, 0x444930FF, 0x464D35FF, 0x353B28FF,
        0x2F3424FF, 0x2B3023FF, 0x282C20FF, 0x24281DFF, 0x20241BFF, 0x1D2018FF, 0x1D2119FF, 0x1E221AFF, 0x1F241CFF, 0x1F241CFF, 0x22281FFF, 0x20251DFF, 0x1D211AFF, 0x191C16FF, 0x141813FF, 0x111410FF,
        0x141206FF, 0x191809FF, 0x1D1B09FF, 0x211E09FF, 0x272109FF, 0x2E280AFF, 0x342E0EFF, 0x383310FF, 0x3A3414FF, 0x393516FF, 0x39371BFF, 0x393A20FF, 0x393C25FF, 0x393E28FF, 0x363B28FF, 0x323626FF,
        0x303526FF, 0x2E3325FF, 0x2B3024FF, 0x292D22FF, 0x262B20FF, 0x21251CFF, 0x1F231AFF, 0x21251DFF, 0x242920FF, 0x272D24FF, 0x242A21FF, 0x242921FF, 0x22271FFF, 0x1E221BFF, 0x181C16FF, 0x131612FF,
        0x110F04FF, 0x141206FF, 0x151305FF, 0x191605FF, 0x1D1907FF, 0x211E08FF, 0x26220AFF, 0x2A260EFF, 0x2C2810FF, 0x2E2C14FF, 0x302F19FF, 0x31321DFF, 0x323522FF, 0x323523FF, 0x313424FF, 0x313626FF,
        0x2F3426FF, 0x303528FF, 0x2D3326FF, 0x2B3226FF, 0x292E24FF, 0x262B21FF, 0x22281FFF, 0x22271FFF, 0x262B22FF, 0x2A3227FF, 0x2A3227FF, 0x232921FF, 0x282F26FF, 0x222820FF, 0x1C211AFF, 0x161A15FF,
        0x0D0C03FF, 0x100E04FF, 0x0F0D02FF, 0x131004FF, 0x161305FF, 0x1A1707FF, 0x1D1A09FF, 0x201D0BFF, 0x23210FFF, 0x282715FF, 0x2C2C1BFF, 0x2C2D1CFF, 0x2C2E1EFF, 0x2E3122FF, 0x2E3323FF, 0x303526FF,
        0x2F3527FF, 0x303629FF, 0x303629FF, 0x2F3529FF, 0x2D3428FF, 0x2A3025FF, 0x272D23FF, 0x262C22FF, 0x282E24FF, 0x2C3329FF, 0x333B30FF, 0x2E352BFF, 0x2F362CFF, 0x282F26FF, 0x21271FFF, 0x1B2019FF,
        0x0A0901FF, 0x0B0901FF, 0xFFFFFFFF, 0x0F0E03FF, 0x110E04FF, 0x141105FF, 0x171508FF, 0x19170AFF, 0x1E1D0FFF, 0x252416FF, 0xFFFFF5FF, 0x2A2C1DFF, 0x2B2E20FF, 0x2C3021FF, 0x303526FF, 0x2F3527FF,
        0x303528FF, 0x303629FF, 0x31382BFF, 0x31382CFF, 0x2C3327FF, 0x2B3227FF, 0x292F25FF, 0x292F25FF, 0x2B3328FF, 0x2D352AFF, 0x3A4437FF, 0x343C31FF, 0x30382DFF, 0x2F382DFF, 0x282F26FF, 0x1F251DFF,
        0x070600FF, 0x080600FF, 0x0A0801FF, 0x0C0A02FF, 0x0E0C03FF, 0x100E04FF, 0x121005FF, 0x141208FF, 0x18170CFF, 0x1F1E13FF, 0xFFFFF7FF, 0x25261AFF, 0x25281CFF, 0x2A2E21FF, 0x2E3426FF, 0x32382AFF,
        0x343A2CFF, 0x31382BFF, 0x32382CFF, 0x333B2EFF, 0x31382CFF, 0x292F25FF, 0x2A3026FF, 0x2D3329FF, 0x2C3429FF, 0x363F33FF, 0x353E32FF, 0x404B3DFF, 0x343D31FF, 0x333C31FF, 0x30392EFF, 0x252C23FF,
        0x060500FF, 0x070500FF, 0x080600FF, 0x0A0802FF, 0x0B0A02FF, 0x0C0B03FF, 0x0E0B04FF, 0x110F06FF, 0x131209FF, 0x16160DFF, 0x1C1C13FF, 0x1C1D14FF, 0x1F2117FF, 0x25281DFF, 0x2B3124FF, 0x2F3528FF,
        0x32392CFF, 0x353D2FFF, 0x343C2FFF, 0x333B2EFF, 0x2B3127FF, 0x2C3328FF, 0x2F362BFF, 0x2E362BFF, 0x313A2EFF, 0x3C4639FF, 0x384235FF, 0x3B4538FF, 0x3D483AFF, 0x3B4639FF, 0x374235FF, 0x2D352BFF,
        0x060400FF, 0x060500FF, 0x060500FF, 0x080601FF, 0x090802FF, 0x0A0902FF, 0x0B0A03FF, 0x0C0B04FF, 0x0F0D06FF, 0x111109FF, 0x14140CFF, 0x171810FF, 0x1A1C13FF, 0x21241AFF, 0x282C21FF, 0x2C3226FF,
        0x2C3126FF, 0x2E3429FF, 0x333B2EFF, 0x2B3327FF, 0x282E24FF, 0x2D3329FF, 0x374033FF, 0x455141FF, 0x4A5747FF, 0x4B5847FF, 0x455142FF, 0x4B5948FF, 0x495747FF, 0x455142FF, 0x404B3DFF, 0x363F33FF,
        0x050400FF, 0x050400FF, 0x060400FF, 0x060500FF, 0x070601FF, 0x080702FF, 0x090802FF, 0x0A0903FF, 0x0B0A04FF, 0x0C0C06FF, 0x10100AFF, 0x14160EFF, 0x181A12FF, 0x1C2017FF, 0x21261CFF, 0x252A20FF,
        0x282E23FF, 0x2B3126FF, 0x2B3327FF, 0x2A3026FF, 0x293026FF, 0x2D3329FF, 0x394336FF, 0x485544FF, 0x5B6A57FF, 0x61725DFF, 0x586754FF, 0x596A56FF, 0x566653FF, 0x505E4DFF, 0x475545FF, 0x3D483BFF,
        0x040300FF, 0x040300FF, 0x050400FF, 0x050400FF, 0x060501FF, 0x060501FF, 0x080602FF, 0x080702FF, 0x090803FF, 0x0A0A05FF, 0x0D0E08FF, 0x14160FFF, 0x191C14FF, 0x1D2119FF, 0x22261DFF, 0x262B21FF,
        0x292E24FF, 0x2E352AFF, 0x323A2EFF, 0x323A2EFF, 0x31382CFF, 0x2D352AFF, 0x364033FF, 0x444F40FF, 0x53604FFF, 0x5B6B57FF, 0x4C5A4AFF, 0x5A6A57FF, 0x52614FFF, 0x495646FF, 0x445142FF, 0x3D483BFF,
        0x040300FF, 0x040300FF, 0x040300FF, 0x040300FF, 0x040300FF, 0x060501FF, 0x060501FF, 0x070602FF, 0x070702FF, 0x090804FF, 0x0B0B07FF, 0x12140EFF, 0x161811FF, 0x191C15FF, 0x20241BFF, 0x282F24FF,
        0x2B3127FF, 0x2F362BFF, 0x374033FF, 0x3D4739FF, 0x3A4436FF, 0x333C30FF, 0x363E33FF, 0x475343FF, 0x4E5B4AFF, 0x4A5747FF, 0x4A5747FF, 0x576654FF, 0x4A5747FF, 0x445142FF, 0x495747FF, 0x30392EFF,
        0x030200FF, 0x030300FF, 0x040300FF, 0x040300FF, 0x040300FF, 0x040300FF, 0x040300FF, 0x050401FF, 0x060502FF, 0x060602FF, 0x0A0A05FF, 0x0F100AFF, 0x12140EFF, 0x171B14FF, 0x1F241CFF, 0x282E24FF,
        0x2F362BFF, 0x343C30FF, 0x374134FF, 0x3E493BFF, 0x404C3DFF, 0x414B3DFF, 0x3B4538FF, 0x455142FF, 0x566653FF, 0x5A6A57FF, 0x52614FFF, 0x60715CFF, 0x5E6F5AFF, 0x4F5D4CFF, 0x232921FF, 0x131512FF,
        0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030300FF, 0x030300FF, 0x030300FF, 0x040501FF, 0x040401FF, 0x040501FF, 0x060603FF, 0x0B0C08FF, 0x12140FFF, 0x1B1E17FF, 0x22271EFF, 0x2B3127FF,
        0x31392DFF, 0x353E31FF, 0x394336FF, 0x3E493AFF, 0x424D3EFF, 0x455041FF, 0x455041FF, 0x445041FF, 0x505E4DFF, 0x586855FF, 0x5C6D59FF, 0x5B6C58FF, 0x536350FF, 0x282F26FF, 0x151913FF, 0x0B0C0AFF,
        0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x040301FF, 0x030200FF, 0x040301FF, 0x050502FF, 0x0C0D09FF, 0x1A1E17FF, 0x21251DFF, 0x282E24FF, 0x2E362BFF,
        0x333D30FF, 0x374134FF, 0x3A4437FF, 0x3E483AFF, 0x414C3EFF, 0x455041FF, 0x475444FF, 0x4A5747FF, 0x4E5B4BFF, 0x576654FF, 0x5B6B58FF, 0x4B5948FF, 0x20261EFF, 0x11140FFF, 0x0A0B09FF, 0x060605FF,
        0x020200FF, 0x020200FF, 0x020200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x030200FF, 0x080804FF, 0x070704FF, 0x0A0B07FF, 0x191D16FF, 0x22281FFF, 0x283026FF, 0x2C3429FF,
        0x323A2FFF, 0x384135FF, 0x3B4538FF, 0x3D4739FF, 0x3F4A3CFF, 0x414D3EFF, 0x445142FF, 0x495646FF, 0x4F5C4CFF, 0xD3E2D0FF, 0x52604FFF, 0xEAF2E9FF, 0x151813FF, 0x0A0A09FF, 0x070705FF, 0x040403FF,
        0x020100FF, 0x020200FF, 0x020200FF, 0x020200FF, 0x020200FF, 0x020200FF, 0x020200FF, 0x020200FF, 0x030301FF, 0x10120DFF, 0x131711FF, 0x181C15FF, 0x1C2119FF, 0x20251DFF, 0x262C23FF, 0x292F26FF,
        0x2E352BFF, 0x353D31FF, 0x3A4336FF, 0x3C4538FF, 0x3C473AFF, 0x3E493BFF, 0x3F4B3DFF, 0x424D3FFF, 0x475344FF, 0x495646FF, 0x414D3EFF, 0x2C342AFF, 0x161914FF, 0x0B0C0AFF, 0x060605FF, 0x030302FF,
        0x030201FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x030401FF, 0x0C0D09FF, 0x11130FFF, 0x131510FF, 0x171A15FF, 0x1F231CFF, 0x20251DFF, 0x1E231CFF, 0x232820FF,
        0x2A3127FF, 0x31392EFF, 0x353F33FF, 0x394437FF, 0x3A4538FF, 0x3C473AFF, 0x3E493BFF, 0x3D483AFF, 0x3E493BFF, 0x3F4A3DFF, 0x3D483BFF, 0x30382EFF, 0x1C1F1AFF, 0x0E0F0CFF, 0x060605FF, 0x040303FF,
        0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x0C0D09FF, 0x0C0D09FF, 0x151712FF, 0x1B1F18FF, 0x171914FF, 0x262C23FF, 0x21261EFF, 0x20241DFF,
        0x272E25FF, 0x323C30FF, 0x343E32FF, 0x353F33FF, 0x353F33FF, 0x364034FF, 0x3B4639FF, 0x3E4A3CFF, 0x384236FF, 0x384235FF, 0x384135FF, 0x31392FFF, 0x222720FF, 0x121411FF, 0x090908FF, 0x060605FF,
        0x010100FF, 0x010100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x020100FF, 0x030201FF, 0x030301FF, 0x090A07FF, 0x131611FF, 0x181B15FF, 0x1B1F19FF, 0x1D201AFF, 0x272E25FF, 0x2B3229FF, 0x323B30FF,
        0x313A2FFF, 0x323B30FF, 0x313A2FFF, 0x343E32FF, 0x353F33FF, 0x353F33FF, 0x363F33FF, 0x3C473AFF, 0x3E4A3CFF, 0x3F4A3DFF, 0x394236FF, 0x343C31FF, 0x292F27FF, 0x171A15FF, 0x0B0C0AFF, 0xFF0907FF,
        0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x020301FF, 0x040503FF, 0x080906FF, 0x0B0D0AFF, 0x151813FF, 0x1A1E18FF, 0x1A1E18FF, 0x1C211BFF, 0x2B3229FF, 0x30382EFF, 0x31392EFF,
        0x2D352BFF, 0x2B3229FF, 0x2F372CFF, 0x323C30FF, 0x374135FF, 0x343D31FF, 0x2F382DFF, 0x2E362CFF, 0x343C31FF, 0x3B4539FF, 0x414C3EFF, 0x3D473BFF, 0x2F372DFF, 0x191D18FF, 0xFF100DFF, 0xFF0A08FF,
        0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x020301FF, 0x060704FF, 0x090B08FF, 0x0E110DFF, 0x131611FF, 0x151813FF, 0x1E231CFF, 0x252C23FF, 0x2C342AFF, 0x1D231CFF, 0x151813FF,
        0x1B2019FF, 0x20261FFF, 0x2A3228FF, 0x30392EFF, 0x353F33FF, 0x363F33FF, 0x30382EFF, 0x272E26FF, 0x2F372DFF, 0x363F34FF, 0x3D473BFF, 0x3E483BFF, 0x333B31FF, 0xFF1E18FF, 0xFF100DFF, 0xFF0908FF,
        0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x020301FF, 0x050604FF, 0x090B08FF, 0x11140FFF, 0x151914FF, 0x181C16FF, 0x232921FF, 0x272E25FF, 0x2D352BFF, 0x21261FFF, 0x151813FF,
        0x181C16FF, 0x191E18FF, 0x232921FF, 0x2A3228FF, 0x343E32FF, 0x353E33FF, 0x2F362CFF, 0x292F27FF, 0x2F362CFF, 0x363F34FF, 0x3B4539FF, 0xFF473AFF, 0xFF382EFF, 0xFF1F19FF, 0xFF100DFF, 0xFF0908FF,
        0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x020201FF, 0x030402FF, 0x080907FF, 0x0E110DFF, 0x161A14FF, 0x1D221BFF, 0x242A22FF, 0x293028FF, 0x2A3228FF, 0x2A3228FF, 0x181D17FF,
        0x1A1F19FF, 0x1E231CFF, 0x21271FFF, 0x272E26FF, 0x2F372DFF, 0x2F372DFF, 0x2B3229FF, 0x2F372DFF, 0x363E33FF, 0x343D32FF, 0xFF4236FF, 0xFF4539FF, 0xFF382EFF, 0xFF211BFF, 0xFF0F0CFF, 0xFF0908FF,
        0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x010100FF, 0x020201FF, 0x030402FF, 0x070805FF, 0x0A0C09FF, 0x10130FFF, 0x191D18FF, 0x242A22FF, 0xEBF2E9FF, 0x2E352CFF, 0x2C342AFF, 0x1E231CFF,
        0x20251EFF, 0x262B24FF, 0x292F27FF, 0x2B3229FF, 0x2F362CFF, 0x30382EFF, 0x374135FF, 0x394337FF, 0x394337FF, 0xFF4337FF, 0xFF4236FF, 0xFF4438FF, 0xFF3C31FF, 0xFF251EFF, 0xFF0D0BFF, 0xFF0706FF,
        0x020101FF, 0x010000FF, 0x010000FF, 0x010100FF, 0x010100FF, 0x020201FF, 0x040503FF, 0x070806FF, 0x0B0D0AFF, 0x10130FFF, 0x151914FF, 0x1D221CFF, 0x262C25FF, 0x2A3128FF, 0x2A3128FF, 0x252B23FF,
        0x262C24FF, 0x2D342BFF, 0x2F372DFF, 0x323B30FF, 0x363F34FF, 0x384136FF, 0x3A4438FF, 0xFF4236FF, 0xFF3F34FF, 0xFF382EFF, 0xFF4337FF, 0xFF3C31FF, 0xFF2E26FF, 0xFF1713FF, 0xFF0C09FF, 0xFF0706FF,
    };
    verify(a, *pix, EXPECT);
}
