/**
  *  \file test/game/vcr/classic/databasetest.cpp
  *  \brief Test for game::vcr::classic::Database
  */

#include "game/vcr/classic/database.hpp"

#include "afl/charset/codepage.hpp"
#include "afl/charset/codepagecharset.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/internalstream.hpp"
#include "afl/test/testrunner.hpp"
#include "game/config/hostconfiguration.hpp"

using afl::base::ConstBytes_t;

/** Verify behaviour of empty database.
    A: default-construct database
    E: expected values produced */
AFL_TEST("game.vcr.classic.Database:init", a)
{
    game::vcr::classic::Database testee;
    a.checkEqual("01. getNumBattles", testee.getNumBattles(), 0U);
    a.checkNull("02. getBattle", testee.getBattle(0));
}


/** Verify loading of empty database.
    A: construct database, load empty VCR.DAT file
    E: expected values produced */
AFL_TEST("game.vcr.classic.Database:load:empty", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        0x00, 0x00
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 0U);

    // Verify saved form
    afl::io::InternalStream out;
    testee.save(out, 0, 0, config, cs);
    a.checkEqualContent<uint8_t>("21. save", out.getContent(), FILE);
}

/** Verify loading of PHost 4 data.
    A: load VCR.DAT file from a PHost4 game
    E: expected values produced. In particular, battles detected as PHost 4. */
AFL_TEST("game.vcr.classic.Database:load:phost4", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        0x02, 0x00, 0x6c, 0x8b, 0x83, 0x33, 0x03, 0x80, 0x01, 0x00, 0xec, 0x01, 0x64, 0x00, 0x56, 0x69,
        0x72, 0x75, 0x73, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x00, 0x00, 0xc7, 0x01, 0x2b, 0x00, 0x08, 0x00, 0x72, 0x48, 0x08, 0x00, 0x08, 0x01,
        0x08, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00, 0x5a, 0x69, 0x6d, 0x70, 0x68, 0x66, 0x69, 0x72,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00,
        0xd8, 0x01, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x7c, 0xab, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x02,
        0x88, 0x01, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x23, 0x54, 0x47, 0x44, 0x42, 0x41, 0x43,
        0x59, 0x56, 0x53, 0x4f, 0x53, 0x51, 0x00, 0x00, 0x2a, 0x03, 0x0b, 0x00, 0x04, 0x00, 0x41, 0x26,
        0x0a, 0x00, 0x0d, 0x01, 0x00, 0x00, 0x0a, 0x00, 0x2f, 0x00, 0x09, 0x00, 0x44, 0x72, 0x61, 0x67,
        0x73, 0x74, 0x65, 0x72, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x00, 0x00, 0x14, 0x01, 0x5a, 0x00, 0x08, 0x00, 0x69, 0x45, 0x09, 0x00, 0x04, 0x01, 0x06, 0x00,
        0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x39, 0x26, 0x3d, 0x24, 0x4a, 0x31,
        0x3f, 0x43, 0x41, 0x3b
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 2U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::PHost4);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 8);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 492);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 8);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 0);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 8);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 4);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), true);
    a.checkEqual("31. getMass",         b1->right().getMass(), 100);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 0);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 0);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(), 0);
    a.checkEqual("35. getCapabilities()", b1->getCapabilities(), 3);

    game::vcr::classic::Battle* b2 = testee.getBattle(1);
    a.checkNonNull("41. getBattle", b2);
    a.checkEqual("42. getType",         b2->getType(), game::vcr::classic::PHost4);
    a.checkEqual("43. getOwner",        b2->left().getOwner(), 4);
    a.checkEqual("44. isPlanet",        b2->left().isPlanet(), false);
    a.checkEqual("45. getMass",         b2->left().getMass(), 526);
    a.checkEqual("46. getNumBeams",     b2->left().getNumBeams(), 13);
    a.checkEqual("47. getNumLaunchers", b2->left().getNumLaunchers(), 9);
    a.checkEqual("48. getNumBays",      b2->left().getNumBays(), 0);
    a.checkEqual("49. getOwner",        b2->right().getOwner(), 8);
    a.checkEqual("50. isPlanet",        b2->right().isPlanet(), false);
    a.checkEqual("51. getMass",         b2->right().getMass(), 392);
    a.checkEqual("52. getNumBeams",     b2->right().getNumBeams(), 4);
    a.checkEqual("53. getNumLaunchers", b2->right().getNumLaunchers(), 0);
    a.checkEqual("54. getNumBays",      b2->right().getNumBays(), 6);
    a.checkEqual("55. getCapabilities()", b2->getCapabilities(), 3);

    // Verify saved form
    afl::io::InternalStream out;
    testee.save(out, 0, 2, config, cs);
    a.checkEqual("61. save size", out.getContent().size(), 202U);
    a.checkEqualContent<uint8_t>("62. save", out.getContent(), ConstBytes_t(FILE).trim(202));
}

/** Verify loading of PHost 3 data.
    A: load VCR.DAT file from a PHost3 game
    E: expected values produced. In particular, battles detected as PHost 3. */
AFL_TEST("game.vcr.classic.Database:load:phost3", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        // join517
        0x01, 0x00, 0x7c, 0x0b, 0x73, 0xb3, 0x00, 0x00, 0x00, 0x00, 0x65, 0x00, 0x18, 0x00, 0x56, 0x41,
        0x4c, 0x49, 0x41, 0x4e, 0x54, 0x20, 0x57, 0x49, 0x4e, 0x44, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x01, 0x27, 0x01, 0x02, 0x00, 0x06, 0x07, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x14, 0x00, 0x04, 0x00, 0x54, 0x2d, 0x31, 0x30, 0x30, 0x30, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x84, 0x00, 0x04, 0x00, 0x56, 0x4d, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x45, 0x42, 0x3a, 0x39, 0x26, 0x3d, 0x43, 0x28, 0x3c, 0x38
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 1U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::PHost3);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 2);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 101);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 2);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 4);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 0);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 4);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), false);
    a.checkEqual("31. getMass",         b1->right().getMass(), 24);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 1);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 0);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(), 0);

    // Verify saved form
    // The original file is not perfectly spec-compliant (null-terminated name).
    static const uint8_t EXPECT[] = {
        0x01, 0x00, 0x7c, 0x0b, 0x73, 0xb3, 0x00, 0x00, 0x00, 0x00, 0x65, 0x00, 0x18, 0x00, 0x56, 0x41,
        0x4c, 0x49, 0x41, 0x4e, 0x54, 0x20, 0x57, 0x49, 0x4e, 0x44, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x09, 0x01, 0x27, 0x01, 0x02, 0x00, 0x06, 0x07, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x14, 0x00, 0x04, 0x00, 0x54, 0x2d, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x01, 0x00,
        0x84, 0x00, 0x04, 0x00, 0x56, 0x4d, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00
    };

    afl::io::InternalStream out;
    testee.save(out, 0, 2, config, cs);
    a.checkEqual("41. save size", out.getContent().size(), 102U);
    a.checkEqualContent<uint8_t>("42. save", out.getContent(), EXPECT);
}

/** Verify loading of PHost 2 data.
    A: load VCR.DAT file from a PHost2 game
    E: expected values produced. In particular, battles detected as PHost 2. */
AFL_TEST("game.vcr.classic.Database:load:phost2", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        0x02, 0x00, 0xe0, 0x24, 0x0f, 0x9a, 0x00, 0x00, 0x00, 0x00, 0x37, 0x00, 0x8c, 0x00, 0x53, 0x48,
        0x49, 0x50, 0x20, 0x20, 0x37, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x07, 0x00, 0x02, 0x00, 0x31, 0x00, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x48, 0x49, 0x50, 0x20, 0x20, 0x39, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbe, 0x00,
        0x09, 0x00, 0x01, 0x00, 0x0a, 0x00, 0x02, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x32, 0x00,
        0x02, 0x00, 0x64, 0x00, 0x64, 0x00, 0x09, 0x49, 0xe6, 0x75, 0x02, 0x0d, 0x40, 0x00, 0x34, 0x00,
        0x02, 0x00, 0x08, 0x00, 0x00, 0x00, 0x2d, 0x00, 0xf4, 0x01, 0x58, 0x02, 0x30, 0x75, 0x00, 0x00,
        0x20, 0x4e, 0x00, 0x00, 0x41, 0x00, 0x64, 0x00, 0x00, 0x00, 0x07, 0x00, 0x14, 0x00, 0x02, 0x00,
        0x02, 0x00, 0x4b, 0x00, 0xeb, 0x00, 0x03, 0x00, 0x13, 0x00, 0x00, 0x00, 0xb8, 0x0b, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x50, 0x00, 0x50, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x36, 0x2d, 0x29, 0x32, 0x3f,
        0x4b, 0x3f, 0x35, 0x2f
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 1U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::PHost2);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 2);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 55);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 2);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 0);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 0);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 1);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), false);
    a.checkEqual("31. getMass",         b1->right().getMass(), 140);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 4);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 2);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(),  0);

    // Verify saved form
    // The config battle magic is rebuilt from scratch; content happens to be same as default
    static const uint8_t EXPECT[] = {
        0x02, 0x00, 0xe0, 0x24, 0x0f, 0x9a, 0x00, 0x00, 0x00, 0x00, 0x37, 0x00, 0x8c, 0x00, 0x53, 0x48,
        0x49, 0x50, 0x20, 0x20, 0x37, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x23, 0x00, 0x07, 0x00, 0x02, 0x00, 0x31, 0x00, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x48, 0x49, 0x50, 0x20, 0x20, 0x39, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0xbe, 0x00,
        0x09, 0x00, 0x01, 0x00, 0x0a, 0x00, 0x02, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x32, 0x00,
        0x02, 0x00, 0x64, 0x00, 0x64, 0x00, 0x0F, 0x0E, 0xE0, 0xB0, 0x02, 0x0F, 0x40, 0x00, 0x34, 0x00,
        0x02, 0x00, 0x08, 0x00, 0x00, 0x00, 0x2d, 0x00, 0xf4, 0x01, 0x58, 0x02, 0x30, 0x75, 0x00, 0x00,
        0x20, 0x4e, 0x00, 0x00, 0x41, 0x00, 0x64, 0x00, 0x00, 0x00, 0x07, 0x00, 0x14, 0x00, 0x02, 0x00,
        0x02, 0x00, 0x4b, 0x00, 0xeb, 0x00, 0x03, 0x00, 0x13, 0x00, 0x00, 0x00, 0xb8, 0x0b, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x50, 0x00, 0x50, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    afl::io::InternalStream out;
    testee.save(out, 0, 99, config, cs);
    a.checkEqual("41. save size", out.getContent().size(), 202U);
    a.checkEqualContent<uint8_t>("42. save", out.getContent(), EXPECT);
}

/** Verify loading of Host data.
    A: load VCR.DAT file from a Host game
    E: expected values produced. In particular, battles detected as Host. */
AFL_TEST("game.vcr.classic.Database:load:host", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        // schule,
        0x01, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x63, 0x00, 0x01, 0x00, 0xb4, 0x00, 0x99, 0x00, 0x43, 0x2e,
        0x43, 0x2e, 0x53, 0x2e, 0x53, 0x2e, 0x20, 0x53, 0x70, 0x61, 0x63, 0x65, 0x6d, 0x61, 0x6e, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x02, 0x01, 0x6a, 0x00, 0x07, 0x00, 0x63, 0x00, 0x06, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x03, 0x00, 0x43, 0x65, 0x73, 0x74, 0x75, 0x73, 0x20, 0x33,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x07, 0x00,
        0x1e, 0x00, 0x09, 0x00, 0x01, 0x00, 0x05, 0x00, 0x04, 0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x48, 0x30, 0x35, 0x41, 0x47, 0x42, 0x3d, 0x32, 0x2f, 0x32
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 1U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::Host);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 7);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 180);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 8);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 3);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 0);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 9);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), true);
    a.checkEqual("31. getMass",         b1->right().getMass(), 153);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 4);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 0);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(), 7);

    // Verify saved form
    // We do not reproduce the "planet temp" attribute
    static const uint8_t EXPECT[] = {
        0x01, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0xb4, 0x00, 0x99, 0x00, 0x43, 0x2e,
        0x43, 0x2e, 0x53, 0x2e, 0x53, 0x2e, 0x20, 0x53, 0x70, 0x61, 0x63, 0x65, 0x6d, 0x61, 0x6e, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x02, 0x01, 0x6a, 0x00, 0x07, 0x00, 0x63, 0x00, 0x06, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x03, 0x00, 0x43, 0x65, 0x73, 0x74, 0x75, 0x73, 0x20, 0x33,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x07, 0x00,
        0x1e, 0x00, 0x09, 0x00, 0x01, 0x00, 0x05, 0x00, 0x04, 0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00
    };

    afl::io::InternalStream out;
    testee.save(out, 0, 99, config, cs);
    a.checkEqual("41. save size", out.getContent().size(), 102U);
    a.checkEqualContent<uint8_t>("42. save", out.getContent(), EXPECT);
}

/** Verify loading of file modified by CORR.
    A: load VCR.DAT file that was modified by CORR.
    E: expected values produced. In particular, the extra battle added by CORR is not listed. */
AFL_TEST("game.vcr.classic.Database:load:corr", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        // join517
        0x02, 0x00, 0x7c, 0x0b, 0x73, 0xb3, 0x00, 0x00, 0x00, 0x00, 0x65, 0x00, 0x18, 0x00, 0x56, 0x41,
        0x4c, 0x49, 0x41, 0x4e, 0x54, 0x20, 0x57, 0x49, 0x4e, 0x44, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x01, 0x27, 0x01, 0x02, 0x00, 0x06, 0x07, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x14, 0x00, 0x04, 0x00, 0x54, 0x2d, 0x31, 0x30, 0x30, 0x30, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x84, 0x00, 0x04, 0x00, 0x56, 0x4d, 0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x3d, 0x3d, 0x3d, 0x3d, 0x20, 0x9d, 0xe2, 0xae, 0x20, 0xad, 0xa5, 0x20, 0xa1, 0xa8,
        0xe2, 0xa2, 0xa0, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0xae, 0xaa, 0xa0,
        0xa7, 0x20, 0xad, 0xa5, 0x20, 0xa2, 0xae, 0xa7, 0xac, 0xae, 0xa6, 0xa5, 0xad, 0x20, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 1U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::PHost3);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 2);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 101);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 2);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 4);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 0);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 4);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), false);
    a.checkEqual("31. getMass",         b1->right().getMass(), 24);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 1);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 0);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(), 0);
}

/** Verify loading of Nu data.
    A: load VCR.DAT file from a Nu game (imported by c2nu).
    E: expected values produced. In particular, battles detected Nuas Host. */
AFL_TEST("game.vcr.classic.Database:load:c2nu", a)
{
    // Environment
    afl::charset::CodepageCharset cs(afl::charset::g_codepage437);
    game::config::HostConfiguration config;

    static const uint8_t FILE[] = {
        0x02, 0x00, 0x0e, 0x00, 0x4e, 0x55, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x78, 0x00, 0x42, 0x32,
        0x30, 0x30, 0x20, 0x31, 0x36, 0x36, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x06, 0x00, 0xa6, 0x00, 0x04, 0x00, 0x01, 0x33, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x49, 0x52, 0x45, 0x43, 0x4c, 0x4f, 0x55,
        0x44, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x53, 0x20, 0x43, 0x52, 0x55, 0x49, 0x00, 0x00, 0xec, 0x00,
        0xa8, 0x00, 0x06, 0x00, 0x01, 0x38, 0x02, 0x00, 0x06, 0x00, 0x00, 0x00, 0x09, 0x00, 0x1e, 0x00,
        0x02, 0x00, 0x64, 0x00, 0x64, 0x00, 0x5c, 0x00, 0x4e, 0x55, 0x07, 0x00, 0x01, 0x00, 0xa5, 0x01,
        0x74, 0x00, 0x54, 0x2d, 0x52, 0x45, 0x58, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x53, 0x20, 0x42, 0x41,
        0x54, 0x54, 0x4c, 0x45, 0x53, 0x48, 0x00, 0x00, 0x2a, 0x03, 0xa4, 0x00, 0x02, 0x00, 0x01, 0x17,
        0x06, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x06, 0x00, 0x77, 0x00, 0x05, 0x00, 0x4c, 0x65, 0x73, 0x68,
        0x79, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x00, 0x00, 0x10, 0x27, 0xc7, 0x00, 0x04, 0x00, 0x01, 0x00, 0x03, 0x00, 0x02, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x64, 0x00, 0x64, 0x00, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
        0x27, 0x28, 0x29, 0x2a
    };
    afl::io::ConstMemoryStream ms(FILE);

    // Action
    game::vcr::classic::Database testee;
    AFL_CHECK_SUCCEEDS(a("01. load"), testee.load(ms, config,cs));

    // Verify
    a.checkEqual("11. getNumBattles", testee.getNumBattles(), 2U);

    game::vcr::classic::Battle* b1 = testee.getBattle(0);
    a.checkNonNull("21. getBattle", b1);
    a.checkEqual("22. getType",         b1->getType(), game::vcr::classic::NuHost);
    a.checkEqual("23. getOwner",        b1->left().getOwner(), 4);
    a.checkEqual("24. isPlanet",        b1->left().isPlanet(), false);
    a.checkEqual("25. getMass",         b1->left().getMass(), 30);
    a.checkEqual("26. getNumBeams",     b1->left().getNumBeams(), 2);
    a.checkEqual("27. getNumLaunchers", b1->left().getNumLaunchers(), 0);
    a.checkEqual("28. getNumBays",      b1->left().getNumBays(), 0);
    a.checkEqual("29. getOwner",        b1->right().getOwner(), 6);
    a.checkEqual("30. isPlanet",        b1->right().isPlanet(), false);
    a.checkEqual("31. getMass",         b1->right().getMass(), 120);
    a.checkEqual("32. getNumBeams",     b1->right().getNumBeams(), 6);
    a.checkEqual("33. getNumLaunchers", b1->right().getNumLaunchers(), 2);
    a.checkEqual("34. getNumBays",      b1->right().getNumBays(), 0);

    game::vcr::classic::Battle* b2 = testee.getBattle(1);
    a.checkNonNull("41. getBattle", b2);
    a.checkEqual("42. getType",         b2->getType(), game::vcr::classic::NuHost);
    a.checkEqual("43. getOwner",        b2->left().getOwner(), 2);
    a.checkEqual("44. isPlanet",        b2->left().isPlanet(), false);
    a.checkEqual("45. getMass",         b2->left().getMass(), 421);
    a.checkEqual("46. getNumBeams",     b2->left().getNumBeams(), 10);
    a.checkEqual("47. getNumLaunchers", b2->left().getNumLaunchers(), 5);
    a.checkEqual("48. getNumBays",      b2->left().getNumBays(), 0);
    a.checkEqual("49. getOwner",        b2->right().getOwner(), 4);
    a.checkEqual("50. isPlanet",        b2->right().isPlanet(), true);
    a.checkEqual("51. getMass",         b2->right().getMass(), 116);
    a.checkEqual("52. getNumBeams",     b2->right().getNumBeams(), 2);
    a.checkEqual("53. getNumLaunchers", b2->right().getNumLaunchers(), 0);
    a.checkEqual("54. getNumBays",      b2->right().getNumBays(), 4);

    // Verify saved form
    // We do not reproduce the "planet temp" attribute. Let's just save one for a change.
    static const uint8_t EXPECT[] = {
        0x01, 0x00, 0x0e, 0x00, 0x4e, 0x55, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x78, 0x00, 0x42, 0x32,
        0x30, 0x30, 0x20, 0x31, 0x36, 0x36, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x00, 0x00, 0x06, 0x00, 0xa6, 0x00, 0x04, 0x00, 0x01, 0x33, 0x02, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x49, 0x52, 0x45, 0x43, 0x4c, 0x4f, 0x55,
        0x44, 0x20, 0x43, 0x4c, 0x41, 0x53, 0x53, 0x20, 0x43, 0x52, 0x55, 0x49, 0x00, 0x00, 0xec, 0x00,
        0xa8, 0x00, 0x06, 0x00, 0x01, 0x38, 0x02, 0x00, 0x06, 0x00, 0x00, 0x00, 0x09, 0x00, 0x1e, 0x00,
        0x02, 0x00, 0x64, 0x00, 0x64, 0x00
    };

    afl::io::InternalStream out;
    testee.save(out, 0, 1, config, cs);
    a.checkEqual("61. save size", out.getContent().size(), 102U);
    a.checkEqualContent<uint8_t>("62. save", out.getContent(), EXPECT);
}
