/**
  *  \file test/game/v3/turnfiletest.cpp
  *  \brief Test for game::v3::TurnFile
  */

#include "game/v3/turnfile.hpp"

#include "afl/base/countof.hpp"
#include "afl/charset/codepage.hpp"
#include "afl/charset/codepagecharset.hpp"
#include "afl/except/fileformatexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/internalstream.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/test/testrunner.hpp"
#include "game/timestamp.hpp"
#include "game/v3/registrationkey.hpp"

using afl::base::ConstBytes_t;
using afl::charset::Charset;
using afl::charset::CodepageCharset;
using afl::except::FileFormatException;
using afl::io::ConstMemoryStream;
using afl::io::InternalStream;
using afl::string::NullTranslator;
using game::Timestamp;
using game::v3::RegistrationKey;
using game::v3::TurnFile;

namespace {
    /* A version 3.5 turn file, made from Tim's example RST using PCC2 */
    const uint8_t TURN35[] = {
        0x0b, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x30, 0x31, 0x2d, 0x31, 0x32, 0x2d, 0x31, 0x39, 0x39, 0x36,
        0x30, 0x32, 0x3a, 0x30, 0x30, 0x3a, 0x32, 0x37, 0x00, 0x00, 0x96, 0x03, 0x00, 0x46, 0x00, 0x00,
        0x00, 0x58, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x8e, 0x00, 0x00,
        0x00, 0xa0, 0x00, 0x00, 0x00, 0xb2, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0xd6, 0x00, 0x00,
        0x00, 0xe8, 0x00, 0x00, 0x00, 0x35, 0x00, 0x8b, 0x00, 0x07, 0x00, 0x09, 0x00, 0x06, 0x00, 0x04,
        0x00, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0x35, 0x00, 0xbd, 0x00, 0x07, 0x00, 0x09, 0x00, 0x0a,
        0x00, 0x04, 0x00, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0x35, 0x00, 0xd1, 0x00, 0x13, 0x00, 0x09,
        0x00, 0x0a, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0x00, 0xe3, 0x00, 0x13,
        0x00, 0x09, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0x00, 0x0d,
        0x01, 0x07, 0x00, 0x09, 0x00, 0x09, 0x00, 0x04, 0x00, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0x35,
        0x00, 0x44, 0x01, 0x13, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x35, 0x00, 0x9e, 0x01, 0x13, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x35, 0x00, 0xb8, 0x01, 0x13, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x35, 0x00, 0xcc, 0x01, 0x07, 0x00, 0x09, 0x00, 0x0a, 0x00, 0x04,
        0x00, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0x35, 0x00, 0xd4, 0x01, 0x07, 0x00, 0x09, 0x00, 0x0a,
        0x00, 0x04, 0x00, 0x09, 0x00, 0x02, 0x00, 0x00, 0x00, 0x56, 0x45, 0x52, 0x33, 0x2e, 0x35, 0x30,
        0x31, 0x62, 0xbd, 0xf9, 0x5e, 0x65, 0xa0, 0x56, 0x6d, 0xf8, 0x3c, 0x32, 0x07, 0xf0, 0x6b, 0xdc,
        0x97, 0x2d, 0x1d, 0x9d, 0x0e, 0x91, 0xbf, 0x6b, 0x09, 0xb6, 0x48, 0x78, 0xeb, 0x0c, 0x7a, 0x82,
        0x93, 0x98, 0xae, 0x7b, 0x73, 0x27, 0xa0, 0x07, 0xbd, 0xf9, 0x48, 0x69, 0xee, 0x2e, 0xe2, 0xd7,
        0x0a, 0x7b, 0xd3, 0x3f, 0x19, 0x99, 0x69, 0x5a, 0xa2, 0xb3, 0xb8, 0xc3, 0xe1, 0x01, 0xe3, 0x43,
        0xe0, 0x78, 0xac, 0xde, 0x96, 0x02, 0x93, 0xb7, 0xe0, 0x3e, 0xf9, 0x80, 0x91, 0x02, 0xc7, 0xd5,
        0xaa, 0xc0, 0xd5, 0x2e, 0x95, 0x84, 0x73, 0x90, 0x2a, 0x8f, 0x16, 0x8c, 0xed, 0xb8, 0x32, 0xa3,
        0x97, 0xc0, 0x1e, 0xd9, 0xa0, 0xb1, 0x22, 0xe7, 0xf5, 0x8a, 0xe0, 0xf5, 0x0e, 0x43, 0x6c, 0x69,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x50, 0x6c, 0x61, 0x6e, 0x65, 0x74, 0x73, 0x20, 0x43, 0x6f, 0x6d,
        0x6d, 0x61, 0x6e, 0x64, 0x20, 0x43, 0x65, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x49, 0x49, 0x20, 0x28,
        0x76, 0x32, 0x2e, 0x30, 0x2e, 0x31, 0x32, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x68,
        0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x70, 0x68, 0x6f, 0x73, 0x74, 0x2e, 0x64, 0x65, 0x2f, 0x7e,
        0x73, 0x74, 0x65, 0x66, 0x61, 0x6e, 0x2f, 0x70, 0x63, 0x63, 0x32, 0x2e, 0x68, 0x74, 0x6d, 0x6c,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x77, 0x00, 0x00, 0x50, 0x43, 0x43, 0x32, 0x5e, 0x04, 0x00,
        0x00, 0x36, 0x07, 0x00, 0x00, 0xe7, 0x09, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x50, 0x14, 0x00,
        0x00, 0xe8, 0x20, 0x00, 0x00, 0x7b, 0x22, 0x00, 0x00, 0xb0, 0x2c, 0x00, 0x00, 0x29, 0x2e, 0x00,
        0x00, 0xe8, 0x3a, 0x00, 0x00, 0x3d, 0x40, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0xeb, 0x4b, 0x00,
        0x00, 0xf0, 0x49, 0x00, 0x00, 0xe3, 0x49, 0x00, 0x00, 0xa0, 0x5c, 0x00, 0x00, 0x31, 0x57, 0x00,
        0x00, 0xc6, 0x6c, 0x00, 0x00, 0x97, 0x5d, 0x00, 0x00, 0xc8, 0x73, 0x00, 0x00, 0xb5, 0x6b, 0x00,
        0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00,
        0x00, 0x5e, 0x04, 0x00, 0x00, 0x42, 0x0a, 0x00, 0x00, 0x5e, 0x11, 0x00, 0x00, 0x5c, 0x17, 0x00,
        0x00, 0xa9, 0x1a, 0x00, 0x00, 0xd2, 0x21, 0x00, 0x00, 0x1a, 0x27, 0x00, 0x00, 0x00, 0x0d, 0x00,
        0x00, 0x4f, 0x17, 0x00, 0x00, 0x5c, 0x17, 0x00, 0x00, 0xd0, 0x1a, 0x00, 0x00, 0x40, 0x1d, 0x00,
        0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00,
        0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00,
        0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00,
        0x00, 0xa0, 0x28, 0x00, 0x00, 0xfb, 0xd5, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x77, 0x00, 0x00
    };

    /* A version 3.0 turn file, made from Tim's example RST using PCC1 */
    const uint8_t TURN30[] = {
        0x0b, 0x00, 0x06, 0x00, 0x00, 0x00, 0x30, 0x31, 0x2d, 0x31, 0x32, 0x2d, 0x31, 0x39, 0x39, 0x36,
        0x30, 0x32, 0x3a, 0x30, 0x30, 0x3a, 0x32, 0x37, 0x00, 0x00, 0x96, 0x03, 0x00, 0x36, 0x00, 0x00,
        0x00, 0x3c, 0x00, 0x00, 0x00, 0x42, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00,
        0x00, 0x54, 0x00, 0x00, 0x00, 0x02, 0x00, 0x5b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x71, 0x00, 0x09,
        0x00, 0x02, 0x00, 0xc5, 0x00, 0x09, 0x00, 0x02, 0x00, 0xcd, 0x00, 0x09, 0x00, 0x02, 0x00, 0x5a,
        0x01, 0x00, 0x00, 0x02, 0x00, 0xad, 0x01, 0x09, 0x00, 0x44, 0x14, 0x00, 0x00, 0x50, 0x43, 0x43,
        0x21, 0x5e, 0x04, 0x00, 0x00, 0x36, 0x07, 0x00, 0x00, 0xe7, 0x09, 0x00, 0x00, 0x80, 0x06, 0x00,
        0x00, 0x50, 0x14, 0x00, 0x00, 0xe8, 0x20, 0x00, 0x00, 0x7b, 0x22, 0x00, 0x00, 0xb0, 0x2c, 0x00,
        0x00, 0x29, 0x2e, 0x00, 0x00, 0xe8, 0x3a, 0x00, 0x00, 0x3d, 0x40, 0x00, 0x00, 0x80, 0x13, 0x00,
        0x00, 0xeb, 0x4b, 0x00, 0x00, 0xf0, 0x49, 0x00, 0x00, 0xe3, 0x49, 0x00, 0x00, 0xa0, 0x5c, 0x00,
        0x00, 0x31, 0x57, 0x00, 0x00, 0xc6, 0x6c, 0x00, 0x00, 0x97, 0x5d, 0x00, 0x00, 0xc8, 0x73, 0x00,
        0x00, 0xb5, 0x6b, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00,
        0x00, 0xa0, 0x28, 0x00, 0x00, 0x5e, 0x04, 0x00, 0x00, 0x42, 0x0a, 0x00, 0x00, 0x5e, 0x11, 0x00,
        0x00, 0x5c, 0x17, 0x00, 0x00, 0xa9, 0x1a, 0x00, 0x00, 0xd2, 0x21, 0x00, 0x00, 0x1a, 0x27, 0x00,
        0x00, 0x00, 0x0d, 0x00, 0x00, 0x4f, 0x17, 0x00, 0x00, 0x5c, 0x17, 0x00, 0x00, 0xd0, 0x1a, 0x00,
        0x00, 0x40, 0x1d, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00,
        0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00,
        0x00, 0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00,
        0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0xfb, 0xd5, 0x07, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x14, 0x00, 0x00
    };

    /* A version 3.0 turn file with attachments */
    const uint8_t TURN30_ATTACHMENT[] = {
        0x4e, 0x43, 0x43, 0x31, 0x37, 0x30, 0x31, 0x41, 0x44, 0x39, 0xdb, 0x00, 0x00, 0x00, 0x59, 0x01,
        0x00, 0x00, 0x34, 0x02, 0x00, 0x00, 0x31, 0x00, 0x00, 0x00, 0x70, 0x63, 0x63, 0x32, 0x2e, 0x69,
        0x6e, 0x69, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x30, 0x31, 0x2d, 0x31, 0x32, 0x2d, 0x31, 0x39, 0x39, 0x36, 0x30, 0x32, 0x3a, 0x30, 0x30, 0x3a,
        0x32, 0x37, 0x00, 0x00, 0x96, 0x03, 0x00, 0x36, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x42,
        0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x02,
        0x00, 0x5b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x71, 0x00, 0x09, 0x00, 0x02, 0x00, 0xc5, 0x00, 0x09,
        0x00, 0x02, 0x00, 0xcd, 0x00, 0x09, 0x00, 0x02, 0x00, 0x5a, 0x01, 0x00, 0x00, 0x02, 0x00, 0xad,
        0x01, 0x09, 0x00, 0x44, 0x14, 0x00, 0x00, 0x50, 0x43, 0x43, 0x21, 0x5e, 0x04, 0x00, 0x00, 0x36,
        0x07, 0x00, 0x00, 0xe7, 0x09, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00, 0x50, 0x14, 0x00, 0x00, 0xe8,
        0x20, 0x00, 0x00, 0x7b, 0x22, 0x00, 0x00, 0xb0, 0x2c, 0x00, 0x00, 0x29, 0x2e, 0x00, 0x00, 0xe8,
        0x3a, 0x00, 0x00, 0x3d, 0x40, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00, 0xeb, 0x4b, 0x00, 0x00, 0xf0,
        0x49, 0x00, 0x00, 0xe3, 0x49, 0x00, 0x00, 0xa0, 0x5c, 0x00, 0x00, 0x31, 0x57, 0x00, 0x00, 0xc6,
        0x6c, 0x00, 0x00, 0x97, 0x5d, 0x00, 0x00, 0xc8, 0x73, 0x00, 0x00, 0xb5, 0x6b, 0x00, 0x00, 0xc0,
        0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x5e,
        0x04, 0x00, 0x00, 0x42, 0x0a, 0x00, 0x00, 0x5e, 0x11, 0x00, 0x00, 0x5c, 0x17, 0x00, 0x00, 0xa9,
        0x1a, 0x00, 0x00, 0xd2, 0x21, 0x00, 0x00, 0x1a, 0x27, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x4f,
        0x17, 0x00, 0x00, 0x5c, 0x17, 0x00, 0x00, 0xd0, 0x1a, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0x20,
        0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0xa0,
        0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00, 0x80, 0x20, 0x00, 0x00, 0x20,
        0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0xa0,
        0x28, 0x00, 0x00, 0xfb, 0xd5, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44,
        0x14, 0x00, 0x00, 0x43, 0x68, 0x61, 0x72, 0x74, 0x2e, 0x47, 0x65, 0x6f, 0x2e, 0x4d, 0x6f, 0x64,
        0x65, 0x20, 0x3d, 0x20, 0x66, 0x6c, 0x61, 0x74, 0x0a, 0x43, 0x68, 0x61, 0x72, 0x74, 0x2e, 0x47,
        0x65, 0x6f, 0x2e, 0x53, 0x69, 0x7a, 0x65, 0x20, 0x3d, 0x20, 0x31, 0x36, 0x30, 0x30, 0x2c, 0x31,
        0x36, 0x30, 0x30, 0x0a
    };

    ConstBytes_t stripLastByte(ConstBytes_t m)
    {
        return m.trim(m.size()-1);
    }

    ConstBytes_t trim(ConstBytes_t m, size_t n)
    {
        return m.trim(n);
    }
}

/** Test initialisation from scratch.
    A: execute "create new turn sequence".
    E: verify correct result produced */
AFL_TEST("game.v3.TurnFile:init", a)
{
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);

    // - construct using player, charset, timestamp
    TurnFile testee(cs, 7, ts);
    a.checkEqual("01. getPlayer",      testee.getPlayer(), 7);
    a.checkEqual("02. getTimestamp",   testee.getTimestamp().getTimestampAsString(), "12-24-200018:30:00");
    a.checkEqual("03. getNumCommands", testee.getNumCommands(), 0U);
    a.checkEqual("04. getFeatures",    testee.getFeatures(), TurnFile::FeatureSet_t(TurnFile::WinplanFeature));
    a.checkEqual("05. charset",       &testee.charset(), &cs);

    // - first set the format (setFeatures, setVersion)
    testee.setFeatures(TurnFile::FeatureSet_t());

    // - then set the registration key (setRegistrationKey)
    RegistrationKey key(std::auto_ptr<Charset>(cs.clone()));
    key.initUnregistered();
    testee.setRegistrationKey(key, 42);

    // - addCommand() can be called anywhere inbetween
    uint8_t arg[] = { 'a','b','c' };
    testee.addCommand(game::v3::tcm_ShipChangeFc, 314, arg);

    // - rebuild the turn (update)
    testee.update();
    a.checkEqual("11. getNumCommands", testee.getNumCommands(), 1U);

    // - write it out.
    InternalStream out;
    testee.write(out);

    // Verify
    // - size is 29 (header) + 4 (command pos) + 7 (command) + 256 (DOS trailer) = 296
    a.checkEqual("21. getSize", out.getSize(), 296U);

    // Verify some content
    ConstBytes_t mem = out.getContent();
    //  +0     WORD    Player Id
    a.checkEqual("31. data", *mem.eat(), 7);
    a.checkEqual("32. data", *mem.eat(), 0);
    //  +2     DWORD   Number of commands. THost accepts no more than 5000.
    a.checkEqual("33. data", *mem.eat(), 1);
    a.checkEqual("34. data", *mem.eat(), 0);
    a.checkEqual("35. data", *mem.eat(), 0);
    a.checkEqual("36. data", *mem.eat(), 0);
    //  +6  10 BYTEs   Date of last turn: mm-dd-yyyy
    // +16   8 BYTEs   Time of last turn: hh:mm:ss
    a.checkEqual("37. data", afl::string::fromBytes(mem.split(18)), "12-24-200018:30:00");
    // +24     WORD    unused
    mem.split(2);
    // +26     WORD    Timestamp Checksum
    a.checkEqual("38. data", *mem.eat(), 133);
    a.checkEqual("39. data", *mem.eat(), 3);
    // --- only, if commands exist ---
    // +28     BYTE    unused
    mem.split(1);
    // +29   n DWORDs  File positions of the commands, each increased by 1
    a.checkEqual("40. data", *mem.eat(), 34);
    a.checkEqual("41. data", *mem.eat(), 0);
    a.checkEqual("42. data", *mem.eat(), 0);
    a.checkEqual("43. data", *mem.eat(), 0);
    // Command:
    //  +0     WORD    Command Code
    //  +2   n BYTEs   Data
    a.checkEqual("44. data", *mem.eat(), 1);
    a.checkEqual("45. data", *mem.eat(), 0);
    a.checkEqual("46. data", *mem.eat(), 314 % 256);
    a.checkEqual("47. data", *mem.eat(), 314 / 256);
    a.checkEqual("48. data", *mem.eat(), 'a');
    a.checkEqual("49. data", *mem.eat(), 'b');
    a.checkEqual("50. data", *mem.eat(), 'c');
}

/** Test parsing a "3.0" turn file.
    A: parse a turn file
    E: verify properties of turn file */
AFL_TEST("game.v3.TurnFile:load:v30", a)
{
    // Setup
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN30);
    TurnFile testee(cs, tx, ms);

    // Parse
    a.checkEqual("01. getTimestamp",        testee.getTimestamp().getTimestampAsString(), "01-12-199602:00:27");
    a.checkEqual("02. getNumCommands",      testee.getNumCommands(), 6U);
    a.checkEqual("03. getPlayer",           testee.getPlayer(), 11);
    a.checkEqual("04. timeChecksum",        testee.getTurnHeader().timeChecksum, 918);
    a.checkEqual("05. getFeatures",         testee.getFeatures(), TurnFile::FeatureSet_t());
    a.checkEqual("06. computeTurnChecksum", testee.computeTurnChecksum(), 0x1444U);

    // Windows trailer and Taccom header is empty
    a.checkEqual("11. getWindowsTrailer", afl::base::fromObject(testee.getWindowsTrailer()).findNot(0),
                 sizeof(game::v3::structures::TurnWindowsTrailer));
    a.checkEqual("12. getTaccomHeader", afl::base::fromObject(testee.getTaccomHeader()).findNot(0),
                 sizeof(game::v3::structures::TaccomTurnHeader));

    // Verify a command
    TurnFile::CommandCode_t code;
    a.check("21. getCommandCode", testee.getCommandCode(2, code));
    a.checkEqual("22. code", code, uint32_t(game::v3::tcm_ShipChangeSpeed));

    int length;
    a.check("31. getCommandLength", testee.getCommandLength(2, length));
    a.checkEqual("32. length", length, 2);

    int id;
    a.check("41. getCommandId", testee.getCommandId(2, id));
    a.checkEqual("42. id", id, 197);

    const char* name = testee.getCommandName(2);
    a.checkNonNull("51. name", name);
    a.checkEqual("52. name", String_t(name), "ShipChangeSpeed");

    ConstBytes_t data = testee.getCommandData(2);
    a.checkGreaterEqual("61. data size", data.size(), 2U);
    a.checkEqual("62. data", *data.eat(), 9);
    a.checkEqual("63. data", *data.eat(), 0);
}

/** Test parsing a "3.0" turn file with attachments.
    A: parse a turn file
    E: verify properties of turn file and attachments */
AFL_TEST("game.v3.TurnFile:load:v30:attachment", a)
{
    // Setup
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN30_ATTACHMENT);
    TurnFile testee(cs, tx, ms);

    // Parse
    a.checkEqual("01. getTimestamp",   testee.getTimestamp().getTimestampAsString(), "01-12-199602:00:27");
    a.checkEqual("02. getNumCommands", testee.getNumCommands(), 6U);
    a.checkEqual("03. getPlayer",      testee.getPlayer(), 11);
    a.checkEqual("04. timeChecksum",   testee.getTurnHeader().timeChecksum, 918);
    a.checkEqual("05. getFeatures",    testee.getFeatures(), TurnFile::FeatureSet_t(TurnFile::TaccomFeature));

    // Verify a command (abridged)
    int id;
    a.check("11. getCommandId", testee.getCommandId(2, id));
    a.checkEqual("12. id", id, 197);

    ConstBytes_t data = testee.getCommandData(2);
    a.checkGreaterEqual("21. data size", data.size(), 2U);
    a.checkEqual("22. data", *data.eat(), 9);
    a.checkEqual("23. data", *data.eat(), 0);

    // Verify attachment
    a.checkEqual("31. getNumFiles", testee.getNumFiles(), 1U);
    a.checkEqual("32. getTaccomTurnPlace", testee.getTaccomTurnPlace(), 0U);
    a.checkEqual("33. name", cs.decode(testee.getTaccomHeader().attachments[0].name), "pcc2.ini");

    // Regenerate
    InternalStream out;
    testee.update();
    testee.write(out);

    // Data must be substantially identical. Since we mess with the 'id' field, it will not be entirely identical, though.
    a.checkEqual("41. getSize", out.getSize(), sizeof(TURN30_ATTACHMENT));
    a.checkEqualContent("42. content", out.getContent().trim(300), afl::base::ConstBytes_t(TURN30_ATTACHMENT).trim(300));
}

/** Test parsing a "3.0" turn file with attachments, header only.
    A: parse a turn file
    E: verify properties of turn file; header-only parsing will ignore attachments */
AFL_TEST("game.v3.TurnFile:load:v30:attachment:header-only", a)
{
    // Setup
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN30_ATTACHMENT);
    TurnFile testee(cs, tx, ms, false);

    // Parse
    a.checkEqual("01. getTimestamp",   testee.getTimestamp().getTimestampAsString(), "01-12-199602:00:27");
    a.checkEqual("02. getNumCommands", testee.getNumCommands(), 0U);
    a.checkEqual("03. getPlayer",      testee.getPlayer(), 11);
    a.checkEqual("04. timeChecksum",   testee.getTurnHeader().timeChecksum, 918);
    a.checkEqual("05. getFeatures",    testee.getFeatures(), TurnFile::FeatureSet_t());
    a.checkEqual("06. getNumFiles",    testee.getNumFiles(), 0U);
}

/** Test parsing a "3.5" turn file.
    A: parse a turn file
    E: verify properties of turn file */
AFL_TEST("game.v3.TurnFile:load:v35", a)
{
    // Setup
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN35);
    TurnFile testee(cs, tx, ms);

    // Parse
    a.checkEqual("01. getTimestamp",        testee.getTimestamp().getTimestampAsString(), "01-12-199602:00:27");
    a.checkEqual("02. getNumCommands",      testee.getNumCommands(), 10U);
    a.checkEqual("03. getPlayer",           testee.getPlayer(), 11);
    a.checkEqual("04. timeChecksum",        testee.getTurnHeader().timeChecksum, 918);
    a.checkEqual("05. getFeatures",         testee.getFeatures(), TurnFile::FeatureSet_t(TurnFile::WinplanFeature));
    a.checkEqual("06. getVersion",          testee.getVersion(), 1);
    a.checkEqual("07. tryGetTurnNr",        testee.tryGetTurnNr(), 117);
    a.checkEqual("08. computeTurnChecksum", testee.computeTurnChecksum(), 0x7701U);

    // Verify a command
    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", testee.getCommandCode(5, code));
    a.checkEqual("12. code", code, uint32_t(game::v3::tcm_BaseBuildShip));

    int length;
    a.check("21. getCommandLength", testee.getCommandLength(5, length));
    a.checkEqual("22. length", length, 14);

    int id;
    a.check("31. getCommandId", testee.getCommandId(5, id));
    a.checkEqual("32. id", id, 324);

    const char* name = testee.getCommandName(5);
    a.checkNonNull("41. name", name);
    a.checkEqual("42. name", String_t(name), "BaseBuildShip");

    ConstBytes_t data = testee.getCommandData(5);
    a.checkGreaterEqual("51. data size", data.size(), 14U);
    a.checkEqual("52. data", *data.eat(), 19);   // hull type
    a.checkEqual("53. data", *data.eat(), 0);
    a.checkEqual("54. data", *data.eat(), 9);    // engine type
    a.checkEqual("55. data", *data.eat(), 0);
    a.checkEqual("56. data", *data.eat(), 10);   // beam type
    a.checkEqual("57. data", *data.eat(), 0);
}

/** Test parsing a "3.5" turn file, header-only mode.
    A: parse a turn file
    E: verify properties of turn file */
AFL_TEST("game.v3.TurnFile:load:v35:header-only", a)
{
    // Setup
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN35);
    TurnFile testee(cs, tx, ms, false);

    // Parse
    a.checkEqual("01. getTimestamp",   testee.getTimestamp().getTimestampAsString(), "01-12-199602:00:27");
    a.checkEqual("02. getNumCommands", testee.getNumCommands(), 0U);
    a.checkEqual("03. getPlayer",      testee.getPlayer(), 11);
    a.checkEqual("04. timeChecksum",   testee.getTurnHeader().timeChecksum, 918);
    a.checkEqual("05. getFeatures",    testee.getFeatures(), TurnFile::FeatureSet_t(TurnFile::WinplanFeature));
    a.checkEqual("06. getVersion",     testee.getVersion(), 1);

    // Command access fails
    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", !testee.getCommandCode(5, code));

    int length;
    a.check("21. getCommandLength", !testee.getCommandLength(5, length));

    int id;
    a.check("31. getCommandId", !testee.getCommandId(5, id));

    const char* name = testee.getCommandName(5);
    a.checkNull("41. name", name);

    ConstBytes_t data = testee.getCommandData(5);
    a.check("51. empty", data.empty());
}

/** Test parsing damaged files.
    A: parse truncated files
    E: parsing fails with FileFormatException */

// Minimum size is 29 + 6*4 + 256 = 309, but otherwise we don't validate that trailer and commands don't overlap
AFL_TEST("game.v3.TurnFile:error:truncated", a)
{
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(trim(TURN30, 308));
    AFL_CHECK_THROWS(a, TurnFile(cs, tx, ms), FileFormatException);
}

// Failure to read (turn) headers
AFL_TEST("game.v3.TurnFile:error:truncated-header", a)
{
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(trim(TURN30, 10));
    AFL_CHECK_THROWS(a, TurnFile(cs, tx, ms), FileFormatException);
}

// With attachments, we can detect a missing last byte
AFL_TEST("game.v3.TurnFile:error:truncated-attachment", a)
{
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(stripLastByte(TURN30_ATTACHMENT));
    AFL_CHECK_THROWS(a, TurnFile(cs, tx, ms), FileFormatException);
}

// Failure to read (attachment) headers
AFL_TEST("game.v3.TurnFile:error:truncated-attachment-header", a)
{
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(trim(TURN30_ATTACHMENT, 10));
    AFL_CHECK_THROWS(a, TurnFile(cs, tx, ms), FileFormatException);
}

/** Test deleteCommand().
    A: load turn file, delete a command, write out again
    E: new turn file must be shorter */
AFL_TEST("game.v3.TurnFile:deleteCommand", a)
{
    // Load a turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN30);
    TurnFile testee(cs, tx, ms);
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 6U);

    // Remove a command
    testee.deleteCommand(3);
    testee.update();

    // Write out again
    InternalStream out;
    testee.write(out);

    // The new file must be 10 bytes shorter (4 bytes index + 6 bytes command)
    a.checkEqual("11. getSize", out.getSize(), ms.getSize() - 10);
    a.checkEqual("12. getNumCommands", testee.getNumCommands(), 5U);
}

/** Test sendTHostAllies().
    A: create new turn file. Use sendTHostAllies().
    E: verify correct commands generated */
AFL_TEST("game.v3.TurnFile:sendTHostAllies", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Send commands
    testee.sendTHostAllies("ff1ee5", 42, "lfm");
    testee.update();

    // Verify
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 3U);
    for (int i = 0; i < 3; ++i) {
        TurnFile::CommandCode_t code;
        a.check("02. getCommandCode", testee.getCommandCode(i, code));
        a.checkEqual("03. code", code, uint32_t(game::v3::tcm_ShipChangeFc));

        int id;
        a.check("11. getCommandId", testee.getCommandId(i, id));
        a.checkEqual("12. id", id, 42);
    }

    ConstBytes_t b = testee.getCommandData(0);
    a.checkEqual("21. data", *b.eat(), 'f');
    a.checkEqual("22. data", *b.eat(), 'f');
    a.checkEqual("23. data", *b.eat(), '1');

    b = testee.getCommandData(1);
    a.checkEqual("31. data", *b.eat(), 'e');
    a.checkEqual("32. data", *b.eat(), 'e');
    a.checkEqual("33. data", *b.eat(), '5');

    b = testee.getCommandData(2);
    a.checkEqual("41. data", *b.eat(), 'l');
    a.checkEqual("42. data", *b.eat(), 'f');
    a.checkEqual("43. data", *b.eat(), 'm');
}

/** Test sendMessageData().
    A: create new turn file. Use sendMessageData().
    E: verify correct command generated */
AFL_TEST("game.v3.TurnFile:sendMessageData", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Send
    static const uint8_t MSG[] = {'z','x','a','b'};
    testee.sendMessageData(7, 3, MSG);

    // Verify
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 1U);

    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", testee.getCommandCode(0, code));
    a.checkEqual("12. code", code, uint32_t(game::v3::tcm_SendMessage));

    int id; // length of message
    a.check("21. getCommandId", testee.getCommandId(0, id));
    a.checkEqual("22. id", id, 4);

    ConstBytes_t b = testee.getCommandData(0);
    a.checkGreaterEqual("31", b.size(), 8U);
    // - from
    a.checkEqual("32. data", *b.eat(), 7);
    a.checkEqual("33. data", *b.eat(), 0);
    // - to
    a.checkEqual("34. data", *b.eat(), 3);
    a.checkEqual("35. data", *b.eat(), 0);
    // - content
    a.checkEqual("36. data", *b.eat(), 'z');
    a.checkEqual("37. data", *b.eat(), 'x');
    a.checkEqual("38. data", *b.eat(), 'a');
    a.checkEqual("39. data", *b.eat(), 'b');
}

/** Test makeShipCommands().
    A: create new turn file. Use makeShipCommands().
    E: verify correct commands generated */
AFL_TEST("game.v3.TurnFile:makeShipCommands", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Define a ship
    game::v3::structures::Ship oldShip;
    game::v3::structures::Ship newShip;
    afl::base::fromObject(oldShip).fill(0);
    afl::base::fromObject(newShip).fill(0);
    oldShip.shipId = 33;
    newShip.shipId = 33;
    oldShip.owner = 7;
    newShip.owner = 7;
    oldShip.mission = 4;
    newShip.mission = 9;
    oldShip.primaryEnemy = 0;
    newShip.primaryEnemy = 2;

    // Encode it
    testee.makeShipCommands(33, oldShip, newShip);

    // Verify
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 2U);

    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", testee.getCommandCode(0, code));
    a.checkEqual("12. code", code, uint32_t(game::v3::tcm_ShipChangeMission));
    a.check("13. getCommandCode", testee.getCommandCode(1, code));
    a.checkEqual("14. code", code, uint32_t(game::v3::tcm_ShipChangePrimaryEnemy));

    // Verify size
    InternalStream out;
    testee.setFeatures(TurnFile::FeatureSet_t());
    testee.update();
    testee.write(out);

    // Expected file size: 29 (header) + 8 (command pos) + 6+6 (command data) + 256 (trailer) = 305
    a.checkEqual("21. getSize", out.getSize(), 305U);
}

/** Test makePlanetCommands().
    A: create new turn file. Use makePlanetCommands().
    E: verify correct commands generated */
AFL_TEST("game.v3.TurnFile:makePlanetCommands", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Define a planet
    game::v3::structures::Planet oldPlanet;
    game::v3::structures::Planet newPlanet;
    afl::base::fromObject(oldPlanet).fill(0);
    afl::base::fromObject(newPlanet).fill(0);
    oldPlanet.planetId = 33;
    newPlanet.planetId = 33;
    oldPlanet.owner = 7;
    newPlanet.owner = 7;
    oldPlanet.numFactories = 1;
    newPlanet.numFactories = 20;
    oldPlanet.buildBaseFlag = 0;
    newPlanet.buildBaseFlag = 1;

    // Encode it
    testee.makePlanetCommands(33, oldPlanet, newPlanet);

    // Verify
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 2U);

    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", testee.getCommandCode(0, code));
    a.checkEqual("12. code", code, uint32_t(game::v3::tcm_PlanetChangeFactories));
    a.check("13. getCommandCode", testee.getCommandCode(1, code));
    a.checkEqual("14. code", code, uint32_t(game::v3::tcm_PlanetBuildBase));

    // Verify size
    InternalStream out;
    testee.setFeatures(TurnFile::FeatureSet_t());
    testee.update();
    testee.write(out);

    // Expected file size: 29 (header) + 8 (command pos) + 6+4 (command data) + 256 (trailer) = 303
    a.checkEqual("21. getSize", out.getSize(), 303U);
}

/** Test makeBaseCommands().
    A: create new turn file. Use makeBaseCommands().
    E: verify correct commands generated */
AFL_TEST("game.v3.TurnFile:makeBaseCommands", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Define a base
    game::v3::structures::Base oldBase;
    game::v3::structures::Base newBase;
    afl::base::fromObject(oldBase).fill(0);
    afl::base::fromObject(newBase).fill(0);
    oldBase.baseId = 33;
    newBase.baseId = 33;
    oldBase.owner = 7;
    newBase.owner = 7;
    oldBase.mission = 0;
    newBase.mission = 5;
    oldBase.engineStorage[0] = 0;
    newBase.engineStorage[0] = 1;

    // Encode it
    testee.makeBaseCommands(33, oldBase, newBase);

    // Verify
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), 2U);

    TurnFile::CommandCode_t code;
    a.check("11. getCommandCode", testee.getCommandCode(0, code));
    a.checkEqual("12. code", code, uint32_t(game::v3::tcm_BaseBuildEngines));
    a.check("13. getCommandCode", testee.getCommandCode(1, code));
    a.checkEqual("14. code", code, uint32_t(game::v3::tcm_BaseChangeMission));

    // Verify size
    InternalStream out;
    testee.setFeatures(TurnFile::FeatureSet_t());
    testee.update();
    testee.write(out);

    // Expected file size: 29 (header) + 8 (command pos) + 6+22 (command data) + 256 (trailer) = 312
    a.checkEqual("21. getSize", out.getSize(), 321U);
}

/** Test modification of turn file header properties.
    A: load a turn file, modify and write out.
    E: new file reports new properties but otherwise same content. */
AFL_TEST("game.v3.TurnFile:header-modification", a)
{
    // Load existing turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    NullTranslator tx;
    ConstMemoryStream ms(TURN35);
    TurnFile testee(cs, tx, ms);

    // Modify and write out again
    testee.setTimestamp(Timestamp(2000, 12, 24, 18, 30, 0));
    testee.setVersion(0);

    InternalStream out;
    testee.update();
    testee.write(out);

    // Size must not change
    a.checkEqual("01. getSize", out.getSize(), ms.getSize());

    // Reload into new instance and compare
    out.setPos(0);
    TurnFile other(cs, tx, out);
    a.checkEqual("11. getTimestamp", other.getTimestamp().getTimestampAsString(), "12-24-200018:30:00");
    a.checkEqual("12. getVersion", other.getVersion(), 0);
}

/** Test sorting.
    A: create a fresh turn file with some commands. Call sortCommands().
    E: verify order */
AFL_TEST("game.v3.TurnFile:sortCommands", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Add some commands:
    static const uint8_t WORD_ARG[] = {0,0};
    static const uint8_t MSG_ARG[] = {'z'};
    testee.addCommand(game::v3::tcm_BaseChangeMission, 50, WORD_ARG);
    testee.addCommand(game::v3::tcm_BaseChangeMission, 60, WORD_ARG);
    testee.addCommand(game::v3::tcm_ShipChangeMission, 1, WORD_ARG);
    testee.addCommand(game::v3::tcm_ShipChangeMission, 100, WORD_ARG);
    testee.addCommand(game::v3::tcm_ShipChangePrimaryEnemy, 1, WORD_ARG);
    testee.addCommand(game::v3::tcm_ShipChangeSpeed, 1, WORD_ARG);
    testee.addCommand(game::v3::tcm_PlanetChangeFactories, 30, WORD_ARG);
    testee.sendMessageData(7, 5, MSG_ARG);
    testee.sendMessageData(7, 6, MSG_ARG);
    testee.addCommand(game::v3::tcm_BaseChangeMission, 70, WORD_ARG);

    // Sort
    testee.sortCommands();

    // Verify
    static const int EXPECT[][2] = {
        { game::v3::tcm_ShipChangeSpeed,        1 },
        { game::v3::tcm_ShipChangeMission,      1 },
        { game::v3::tcm_ShipChangePrimaryEnemy, 1 },
        { game::v3::tcm_ShipChangeMission,    100 },
        { game::v3::tcm_PlanetChangeFactories, 30 },
        { game::v3::tcm_BaseChangeMission,     50 },
        { game::v3::tcm_BaseChangeMission,     60 },
        { game::v3::tcm_BaseChangeMission,     70 },
        { game::v3::tcm_SendMessage,            1 },
        { game::v3::tcm_SendMessage,            1 },
    };
    a.checkEqual("01. getNumCommands", testee.getNumCommands(), countof(EXPECT));
    for (size_t i = 0; i < countof(EXPECT); ++i) {
        TurnFile::CommandCode_t cmd;
        a.check("02. getCommandCode", testee.getCommandCode(i, cmd));
        a.checkEqual("03. code", int(cmd), EXPECT[i][0]);

        int id;
        a.check("11. getCommandId", testee.getCommandId(i, id));
        a.checkEqual("12. id", id, EXPECT[i][1]);
    }
}

/** Test setRegistrationKey().
    (This is a regression test mainly aimed at ports.)
    A: create a fresh turn file. Call setRegistrationKey() and write it out.
    E: verify against reference data */
AFL_TEST("game.v3.TurnFile:setRegistrationKey", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Set key
    RegistrationKey key(std::auto_ptr<Charset>(cs.clone()));
    key.initUnregistered();
    key.setLine(RegistrationKey::Line3, "Name");
    key.setLine(RegistrationKey::Line4, "Address");
    testee.setRegistrationKey(key, 500);

    // Write out
    InternalStream out;
    testee.update();
    testee.write(out);

    // Verify
    static const uint8_t REFERENCE[] = {
        0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0x32, 0x2d, 0x32, 0x34, 0x2d, 0x32, 0x30, 0x30, 0x30,
        0x31, 0x38, 0x3a, 0x33, 0x30, 0x3a, 0x30, 0x30, 0x00, 0x00, 0x85, 0x03, 0x56, 0x45, 0x52, 0x33,
        0x2e, 0x35, 0x30, 0x31, 0x47, 0x08, 0xaf, 0x1a, 0xbf, 0x05, 0x03, 0x62, 0x47, 0x60, 0x65, 0x96,
        0xa7, 0xa5, 0x15, 0x33, 0x5d, 0x07, 0x3f, 0xab, 0xb6, 0x92, 0x68, 0x9c, 0x5e, 0x65, 0x21, 0xf0,
        0x1a, 0xc6, 0xf9, 0xca, 0x1c, 0x11, 0x27, 0x24, 0xb6, 0xf7, 0xc9, 0x74, 0x5d, 0x38, 0x73, 0x4c,
        0x8b, 0xc5, 0xfa, 0x09, 0xee, 0x3b, 0x12, 0x40, 0x82, 0x7f, 0xe6, 0xd9, 0xea, 0x3c, 0xb2, 0x7f,
        0x26, 0x6d, 0x01, 0x2c, 0x21, 0x5b, 0x76, 0xe5, 0x21, 0x08, 0x22, 0x17, 0xbf, 0x9f, 0x89, 0xab,
        0xb1, 0x2f, 0x0e, 0x6d, 0x28, 0xfc, 0x79, 0xe2, 0x3c, 0x65, 0x4d, 0x48, 0x65, 0x01, 0x7b, 0x56,
        0xc5, 0x01, 0x28, 0x02, 0x37, 0x9f, 0xbf, 0xa9, 0x8b, 0x91, 0x0f, 0x2e, 0x4d, 0x08, 0xdc, 0x59,
        0x4e, 0x61, 0x6d, 0x65, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x4d, 0x00, 0x00, 0x43, 0x32, 0x4e, 0x47,
        0x5e, 0x04, 0x00, 0x00, 0x36, 0x07, 0x00, 0x00, 0xe7, 0x09, 0x00, 0x00, 0x80, 0x06, 0x00, 0x00,
        0x50, 0x14, 0x00, 0x00, 0xe8, 0x20, 0x00, 0x00, 0x7b, 0x22, 0x00, 0x00, 0xb0, 0x2c, 0x00, 0x00,
        0x29, 0x2e, 0x00, 0x00, 0xe8, 0x3a, 0x00, 0x00, 0x3d, 0x40, 0x00, 0x00, 0x80, 0x13, 0x00, 0x00,
        0xeb, 0x4b, 0x00, 0x00, 0xf0, 0x49, 0x00, 0x00, 0xe3, 0x49, 0x00, 0x00, 0xa0, 0x5c, 0x00, 0x00,
        0x31, 0x57, 0x00, 0x00, 0xc6, 0x6c, 0x00, 0x00, 0x97, 0x5d, 0x00, 0x00, 0xc8, 0x73, 0x00, 0x00,
        0xb5, 0x6b, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00,
        0xa0, 0x28, 0x00, 0x00, 0x10, 0x04, 0x00, 0x00, 0xce, 0x06, 0x00, 0x00, 0x35, 0x0a, 0x00, 0x00,
        0x80, 0x06, 0x00, 0x00, 0x89, 0x12, 0x00, 0x00, 0x3e, 0x16, 0x00, 0x00, 0x60, 0x0b, 0x00, 0x00,
        0x00, 0x0d, 0x00, 0x00, 0xa0, 0x0e, 0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xe0, 0x11, 0x00, 0x00,
        0x80, 0x13, 0x00, 0x00, 0x20, 0x15, 0x00, 0x00, 0xc0, 0x16, 0x00, 0x00, 0x60, 0x18, 0x00, 0x00,
        0x00, 0x1a, 0x00, 0x00, 0xa0, 0x1b, 0x00, 0x00, 0x40, 0x1d, 0x00, 0x00, 0xe0, 0x1e, 0x00, 0x00,
        0x80, 0x20, 0x00, 0x00, 0x20, 0x22, 0x00, 0x00, 0xc0, 0x23, 0x00, 0x00, 0x60, 0x25, 0x00, 0x00,
        0x00, 0x27, 0x00, 0x00, 0xa0, 0x28, 0x00, 0x00, 0x4b, 0x68, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    a.checkEqual("01. size", out.getContent().size(), sizeof(REFERENCE));
    a.checkEqualContent("02. content", out.getContent(), afl::base::ConstBytes_t(REFERENCE));
}

/** Test adding files.
    A: add a sequence of addFile(), deleteFile()
    E: verify results */
AFL_TEST("game.v3.TurnFile:addFile", a)
{
    // Empty turn file
    CodepageCharset cs(afl::charset::g_codepage437);
    Timestamp ts(2000, 12, 24, 18, 30, 0);
    TurnFile testee(cs, 7, ts);

    // Add 5 files
    static const uint8_t DATA[] = {'x'};
    size_t pos;
    a.check     ("01. addFile", testee.addFile(DATA, "one", pos));
    a.checkEqual("02. position", pos, 0U);
    a.checkEqual("03. getFeatures", testee.getFeatures(), TurnFile::FeatureSet_t() + TurnFile::TaccomFeature + TurnFile::WinplanFeature);
    a.check     ("04. addFile", testee.addFile(DATA, "two", pos));
    a.checkEqual("05. position", pos, 1U);
    a.check     ("06. addFile", testee.addFile(DATA, "three", pos));
    a.checkEqual("07. position", pos, 2U);
    a.check     ("08. addFile", testee.addFile(DATA, "four", pos));
    a.checkEqual("09. position", pos, 3U);
    a.check     ("10. addFile", testee.addFile(DATA, "five", pos));
    a.checkEqual("11. position", pos, 4U);
    a.checkEqual("12. getNumFiles", testee.getNumFiles(), 5U);

    // Remove one file
    testee.deleteFile(2);
    a.checkEqual("21. getNumFiles", testee.getNumFiles(), 4U);

    // Add one; this takes the free space
    a.check     ("31. addFile", testee.addFile(DATA, "six", pos));
    a.checkEqual("32. position", pos, 2U);

    // Add more
    a.check     ("41. addFile", testee.addFile(DATA, "seven", pos));
    a.checkEqual("42. position", pos, 5U);
    a.check     ("43. addFile", testee.addFile(DATA, "eight", pos));
    a.checkEqual("44. position", pos, 6U);
    a.check     ("45. addFile", testee.addFile(DATA, "nine", pos));
    a.checkEqual("46. position", pos, 7U);
    a.check     ("47. addFile", testee.addFile(DATA, "ten", pos));
    a.checkEqual("48. position", pos, 8U);
    a.check     ("49. addFile", testee.addFile(DATA, "eleven", pos));
    a.checkEqual("50. position", pos, 9U);
    a.checkEqual("51. getNumFiles", testee.getNumFiles(), 10U);

    // It's now full
    a.check("61. addFile", !testee.addFile(DATA, "twelve", pos));
}

/** Test static command property access methods. */
AFL_TEST("game.v3.TurnFile:statics", a)
{
    // Generic
    for (int i = 0; i < 1000; ++i) {
        if (TurnFile::getCommandCodeType(i) != TurnFile::UndefinedCommand) {
            a.checkNonNull("01. getCommandCodeName", TurnFile::getCommandCodeName(i));
            if (TurnFile::getCommandCodeType(i) != TurnFile::OtherCommand) {
                a.checkDifferent("02. getCommandCodeRecordIndex", TurnFile::getCommandCodeRecordIndex(i), 0U);
            }
        }
    }

    // Specific
    a.checkEqual("11", TurnFile::getCommandCodeType(game::v3::tcm_SendMessage), TurnFile::OtherCommand);
    a.checkEqual("12", String_t(TurnFile::getCommandCodeName(game::v3::tcm_SendMessage)), "SendMessage");

    a.checkEqual("21", TurnFile::getCommandCodeType(game::v3::tcm_ShipChangeWaypoint), TurnFile::ShipCommand);
    a.checkEqual("22", String_t(TurnFile::getCommandCodeName(game::v3::tcm_ShipChangeWaypoint)), "ShipChangeWaypoint");

    a.checkEqual("31", TurnFile::getCommandCodeType(game::v3::tcm_PlanetChangeFactories), TurnFile::PlanetCommand);
    a.checkEqual("32", String_t(TurnFile::getCommandCodeName(game::v3::tcm_PlanetChangeFactories)), "PlanetChangeFactories");

    a.checkEqual("41", TurnFile::getCommandCodeType(game::v3::tcm_BaseChangeDefense), TurnFile::BaseCommand);
    a.checkEqual("42", String_t(TurnFile::getCommandCodeName(game::v3::tcm_BaseChangeDefense)), "BaseChangeDefense");
}
