/**
  *  \file test/game/score/loadertest.cpp
  *  \brief Test for game::score::Loader
  */

#include "game/score/loader.hpp"

#include "afl/charset/utf8charset.hpp"
#include "afl/except/fileproblemexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/internalstream.hpp"
#include "afl/string/nulltranslator.hpp"
#include "afl/test/testrunner.hpp"
#include "game/score/turnscore.hpp"
#include "game/score/turnscorelist.hpp"

/**
  *  \file u/t_game_score_loader.cpp
  *  \brief Test for game::score::Loader
  */



namespace {
    // File content (from a test game, p119deb)
    const uint8_t OLD_FILE[] = {
        0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0x02, 0x00, 0x82, 0x00,
        0x03, 0x00, 0x30, 0x34, 0x2d, 0x32, 0x33, 0x2d, 0x32, 0x30, 0x30, 0x39,
        0x30, 0x35, 0x3a, 0x31, 0x31, 0x3a, 0x30, 0x35, 0x05, 0x00, 0x02, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x26, 0x00,
        0x31, 0x31, 0x2d, 0x30, 0x35, 0x2d, 0x32, 0x30, 0x30, 0x39, 0x30, 0x35,
        0x3a, 0x31, 0x30, 0x3a, 0x35, 0x33, 0x0b, 0x00, 0x05, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x2b, 0x00, 0x41, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x3c, 0x00,
        0x2f, 0x00, 0x15, 0x00, 0x09, 0x00, 0x44, 0x00, 0x20, 0x00, 0x17, 0x00,
        0x12, 0x00, 0x2f, 0x00, 0x1f, 0x00, 0x0f, 0x00, 0x08, 0x00, 0x41, 0x00,
        0x2d, 0x00, 0x12, 0x00, 0x0c, 0x00, 0x2b, 0x00, 0x2d, 0x00, 0x18, 0x00,
        0x04, 0x00, 0x3e, 0x00, 0x19, 0x00, 0x15, 0x00, 0x0f, 0x00, 0x19, 0x00,
        0x0d, 0x00, 0x07, 0x00, 0x04, 0x00, 0x36, 0x00, 0x20, 0x00, 0x12, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x5a, 0x04, 0x00, 0x00, 0x00, 0x00
    };
}


/** Test a simple file. */
AFL_TEST("game.score.Loader:simple", a)
{
    // planets/c2nu/t2/score.cc:
    //   a score file from c2nu, containing turn 7 and 8, with identical scores.
    static const uint8_t FILE[] = {
        0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30, 0x1a, 0xd6, 0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00,
        0x14, 0x00, 0x07, 0x00, 0x18, 0x00, 0x26, 0x00, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xfc, 0xff,
        0x02, 0x00, 0xe8, 0x03, 0xe9, 0x03, 0x03, 0x00, 0x4d, 0x69, 0x6c, 0x69, 0x74, 0x61, 0x72, 0x79,
        0x20, 0x53, 0x63, 0x6f, 0x72, 0x65, 0x20, 0x28, 0x4e, 0x75, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xe8, 0x03, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0x49, 0x6e, 0x76, 0x65, 0x6e, 0x74, 0x6f, 0x72, 0x79, 0x20, 0x53, 0x63, 0x6f, 0x72,
        0x65, 0x20, 0x28, 0x4e, 0x75, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0xe9, 0x03, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x42, 0x75, 0x69, 0x6c,
        0x64, 0x20, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x20, 0x28, 0x4e, 0x75, 0x29, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x02, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x07, 0x00, 0x30, 0x35, 0x2d, 0x32, 0x31, 0x2d, 0x32, 0x30,
        0x31, 0x36, 0x30, 0x39, 0x3a, 0x30, 0x33, 0x3a, 0x35, 0x39, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x30, 0x06, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x55, 0x24,
        0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0x1f, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbf, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x08, 0x00,
        0x30, 0x35, 0x2d, 0x32, 0x31, 0x2d, 0x32, 0x30, 0x31, 0x36, 0x30, 0x39, 0x3a, 0x30, 0x33, 0x3a,
        0x35, 0x39, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x67, 0x06,
        0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x32, 0x23, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x00, 0x00, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xbf, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff
    };

    // TurnScoreList:
    game::score::TurnScoreList list;

    // Load it
    {
        afl::string::NullTranslator tx;
        afl::charset::Utf8Charset cs;
        afl::io::ConstMemoryStream ms(FILE);
        AFL_CHECK_SUCCEEDS(a("01. load"), game::score::Loader(tx, cs).load(list, ms));
    }

    // Verify it
    a.checkEqual("11. getNumTurns", list.getNumTurns(), 2U);
    a.checkEqual("12. getNumScores", list.getNumScores(), 7U);       // 4 default scores, PBPs, Military, Inventory
    a.checkEqual("13. getNumDescriptions", list.getNumDescriptions(), 3U); // PBPs, Military, Inventory
    a.checkEqual("14. getTurnNumber", list.getTurnByIndex(0)->getTurnNumber(), 7);
    a.checkEqual("15. getTurnNumber", list.getTurnByIndex(1)->getTurnNumber(), 8);

    game::score::TurnScore::Slot_t planetSlot = 0, freighterSlot = 0, capitalSlot = 0, baseSlot = 0, pbpSlot = 0, milSlot = 0;
    a.check("21. planetSlot",    list.getSlot(game::score::ScoreId_Planets)    .get(planetSlot));
    a.check("22. freighterSlot", list.getSlot(game::score::ScoreId_Freighters) .get(freighterSlot));
    a.check("23. capitalSlot",   list.getSlot(game::score::ScoreId_Capital)    .get(capitalSlot));
    a.check("24. baseSlot",      list.getSlot(game::score::ScoreId_Bases)      .get(baseSlot));
    a.check("25. pbpSlot",       list.getSlot(game::score::ScoreId_BuildPoints).get(pbpSlot));
    a.check("26. milSlot",       list.getSlot(1000)                            .get(milSlot));

    // Verify scores
    const game::score::TurnScore* p = list.getTurn(7);
    a.checkNonNull("31. getTurn", p);
    a.checkEqual("32. planetSlot",    p->get(planetSlot,    1).orElse(-1), 3);
    a.checkEqual("33. freighterSlot", p->get(freighterSlot, 1).orElse(-1), 1);
    a.checkEqual("34. capitalSlot",   p->get(capitalSlot,   1).orElse(-1), 0);
    a.checkEqual("35. baseSlot",      p->get(baseSlot,      1).orElse(-1), 0);
    a.checkEqual("36. pbpSlot",       p->get(pbpSlot,       1).orElse(-1), 0);
    a.checkEqual("37. milSlot",       p->get(milSlot,       1).orElse(-1), 1584);
    a.checkEqual("38. planetSlot",    p->get(planetSlot,    7).orElse(-1), 5);
    a.checkEqual("39. freighterSlot", p->get(freighterSlot, 7).orElse(-1), 1);
    a.checkEqual("40. capitalSlot",   p->get(capitalSlot,   7).orElse(-1), 2);
    a.checkEqual("41. baseSlot",      p->get(baseSlot,      7).orElse(-1), 1);
    a.checkEqual("42. pbpSlot",       p->get(pbpSlot,       7).orElse(-1), 0);
    a.checkEqual("43. milSlot",       p->get(milSlot,       7).orElse(-1), 9301);

    p = list.getTurn(8);
    a.checkNonNull("51. getTurn", p);
    a.checkEqual("52. planetSlot",    p->get(planetSlot,    1).orElse(-1), 3);
    a.checkEqual("53. freighterSlot", p->get(freighterSlot, 1).orElse(-1), 1);
    a.checkEqual("54. capitalSlot",   p->get(capitalSlot,   1).orElse(-1), 0);
    a.checkEqual("55. baseSlot",      p->get(baseSlot,      1).orElse(-1), 0);
    a.checkEqual("56. pbpSlot",       p->get(pbpSlot,       1).orElse(-1), 0);
    a.checkEqual("57. milSlot",       p->get(milSlot,       1).orElse(-1), 1639);
    a.checkEqual("58. planetSlot",    p->get(planetSlot,    7).orElse(-1), 5);
    a.checkEqual("59. freighterSlot", p->get(freighterSlot, 7).orElse(-1), 1);
    a.checkEqual("60. capitalSlot",   p->get(capitalSlot,   7).orElse(-1), 2);
    a.checkEqual("61. baseSlot",      p->get(baseSlot,      7).orElse(-1), 1);
    a.checkEqual("62. pbpSlot",       p->get(pbpSlot,       7).orElse(-1), 0);
    a.checkEqual("63. milSlot",       p->get(milSlot,       7).orElse(-1), 9010);

    // Verify descriptions
    const game::score::TurnScoreList::Description* desc = list.getDescription(1000);
    a.checkNonNull("71", desc);
    a.checkEqual("72. name",      desc->name, "Military Score (Nu)");
    a.checkEqual("73. scoreId",   desc->scoreId, 1000);
    a.checkEqual("74. turnLimit", desc->turnLimit, -1);
    a.checkEqual("75. winLimit",  desc->winLimit, -1);

    desc = list.getDescription(1001);
    a.checkNonNull("81", desc);
    a.checkEqual("82. name",      desc->name, "Inventory Score (Nu)");
    a.checkEqual("83. scoreId",   desc->scoreId, 1001);
    a.checkEqual("84. turnLimit", desc->turnLimit, -1);
    a.checkEqual("85. winLimit",  desc->winLimit, -1);
}

/*
 *  Test some errors
 */

// Truncated file
AFL_TEST("game.score.Loader:load:error:truncated", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = {
        0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30, 0x1a, 0xd6, 0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00,
        0x14, 0x00, 0x07, 0x00, 0x18, 0x00, 0x26, 0x00, 0xff, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0xfc, 0xff,
        0x02, 0x00, 0xe8, 0x03, 0xe9, 0x03, 0x03, 0x00, 0x4d, 0x69, 0x6c, 0x69, 0x74, 0x61, 0x72, 0x79,
        0x20, 0x53, 0x63, 0x6f, 0x72, 0x65, 0x20, 0x28, 0x4e, 0x75, 0x29, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xe8, 0x03, 0xff, 0xff, 0xff, 0xff,
        0xff, 0xff, 0x49, 0x6e, 0x76, 0x65, 0x6e, 0x74, 0x6f, 0x72, 0x79, 0x20, 0x53, 0x63, 0x6f, 0x72,
    };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.load(list, ms), afl::except::FileProblemException);
}

// Truncated signature
AFL_TEST("game.score.Loader:load:error:truncated-signature", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = { 0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30 };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.load(list, ms), afl::except::FileProblemException);
}

// Truncated header
AFL_TEST("game.score.Loader:load:error:truncated-header", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = { 0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x30, 0x1a, 0xd6, 0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00 };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.load(list, ms), afl::except::FileProblemException);
}

// Future file
AFL_TEST("game.score.Loader:load:error:future-file", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = { 0x43, 0x43, 0x73, 0x74, 0x61, 0x74, 0x39, 0x1a, 0xd6, 0x00, 0x00, 0x00, 0x02, 0x00, 0x02, 0x00 };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.load(list, ms), afl::except::FileProblemException);
}


/** Test loadOldFile(). */
AFL_TEST("game.score.Loader:loadOldFile", a)
{
    // TurnScoreList:
    game::score::TurnScoreList list;

    // Load it
    {
        afl::string::NullTranslator tx;
        afl::charset::Utf8Charset cs;
        afl::io::ConstMemoryStream ms(OLD_FILE);
        AFL_CHECK_SUCCEEDS(a("01. loadOldFile"), game::score::Loader(tx, cs).loadOldFile(list, ms));
    }

    // Verify
    a.checkEqual("11. getNumTurns", list.getNumTurns(), 2U);
    a.checkEqual("12. getNumScores", list.getNumScores(), 5U);   // pla/cap/fre/bas/pbp
    a.checkEqual("13. getNumDescriptions", list.getNumDescriptions(), 0U);
    a.checkEqual("14. getTurnNumber", list.getTurnByIndex(0)->getTurnNumber(), 3);
    a.checkEqual("15. getTurnNumber", list.getTurnByIndex(1)->getTurnNumber(), 38);

    // Slots
    game::score::TurnScore::Slot_t planetSlot = 0, freighterSlot = 0, capitalSlot = 0, baseSlot = 0, pbpSlot = 0;
    a.check("21. planetSlot",    list.getSlot(game::score::ScoreId_Planets)    .get(planetSlot));
    a.check("22. freighterSlot", list.getSlot(game::score::ScoreId_Freighters) .get(freighterSlot));
    a.check("23. capitalSlota",  list.getSlot(game::score::ScoreId_Capital)    .get(capitalSlot));
    a.check("24. baseSlot",      list.getSlot(game::score::ScoreId_Bases)      .get(baseSlot));
    a.check("25. pbpSlot",       list.getSlot(game::score::ScoreId_BuildPoints).get(pbpSlot));

    // Verify scores
    const game::score::TurnScore* p = list.getTurn(3);
    a.checkNonNull("31. getTurn", p);
    a.checkEqual("32. planetSlot",    p->get(planetSlot,    1).orElse(-1), 5);
    a.checkEqual("33. freighterSlot", p->get(freighterSlot, 1).orElse(-1), 2);
    a.checkEqual("34. capitalSlot",   p->get(capitalSlot,   1).orElse(-1), 2);
    a.checkEqual("35. baseSlot",      p->get(baseSlot,      1).orElse(-1), 1);
    a.checkEqual("36. pbpSlot",       p->get(pbpSlot,       1).orElse(-1), 0);

    a.checkEqual("41. planetSlot",    p->get(planetSlot,    7).orElse(-1), 4);
    a.checkEqual("42. freighterSlot", p->get(freighterSlot, 7).orElse(-1), 4);
    a.checkEqual("43. capitalSlot",   p->get(capitalSlot,   7).orElse(-1), 0);
    a.checkEqual("44. baseSlot",      p->get(baseSlot,      7).orElse(-1), 1);
    a.checkEqual("45. pbpSlot",       p->get(pbpSlot,       7).orElse(-1), 0);

    p = list.getTurn(38);
    a.checkNonNull("51. getTurn", p);
    a.checkEqual("52. planetSlot",    p->get(planetSlot,    9).orElse(-1), 25);
    a.checkEqual("53. freighterSlot", p->get(freighterSlot, 9).orElse(-1), 7);
    a.checkEqual("54. capitalSlot",   p->get(capitalSlot,   9).orElse(-1), 13);
    a.checkEqual("55. baseSlot",      p->get(baseSlot,      9).orElse(-1), 4);
    a.checkEqual("56. pbpSlot",       p->get(pbpSlot,       9).orElse(-1), 1114);
}


// Truncated file
AFL_TEST("game.score.Loader:loadOldFile:error:truncated", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = {
        0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0x02, 0x00, 0x82, 0x00,
        0x03, 0x00, 0x30, 0x34, 0x2d, 0x32, 0x33, 0x2d, 0x32, 0x30, 0x30, 0x39,
        0x30, 0x35, 0x3a, 0x31, 0x31, 0x3a, 0x30, 0x35, 0x05, 0x00, 0x02, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
    };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.loadOldFile(list, ms), afl::except::FileProblemException);
}

// Truncated signature
AFL_TEST("game.score.Loader:loadOldFile:error:truncated-signature", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = { 0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74 };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.loadOldFile(list, ms), afl::except::FileProblemException);
}

// Truncated header
AFL_TEST("game.score.Loader:loadOldFile:error:truncated-header", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = { 0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0x02, 0x00, 0x82 };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.loadOldFile(list, ms), afl::except::FileProblemException);
}

// Size too small
AFL_TEST("game.score.Loader:loadOldFile:error:size-too-small", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = {                              // vvvv too small
        0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0x02, 0x00, 0x80, 0x00,
        0x03, 0x00, 0x30, 0x34, 0x2d, 0x32, 0x33, 0x2d, 0x32, 0x30, 0x30, 0x39,
    };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.loadOldFile(list, ms), afl::except::FileProblemException);
}

// Bad count
AFL_TEST("game.score.Loader:loadOldFile:error:bad-count", a)
{
    afl::string::NullTranslator tx;
    afl::charset::Utf8Charset cs;
    game::score::Loader testee(tx, cs);
    static const uint8_t FILE[] = {                  // vvvvvvvvvv negative
        0x43, 0x43, 0x2d, 0x53, 0x74, 0x61, 0x74, 0x1a, 0xff, 0xff, 0x82, 0x00,
        0x03, 0x00, 0x30, 0x34, 0x2d, 0x32, 0x33, 0x2d, 0x32, 0x30, 0x30, 0x39,
    };
    game::score::TurnScoreList list;
    afl::io::ConstMemoryStream ms(FILE);
    AFL_CHECK_THROWS(a, testee.loadOldFile(list, ms), afl::except::FileProblemException);
}


/** Test save(). */
AFL_TEST("game.score.Loader:save", a)
{
    // Populate a TurnScoreList and save it.
    // Reload and check that it's still the same.
    afl::io::InternalStream s;

    // Load & Save it
    {
        game::score::TurnScoreList list;
        afl::string::NullTranslator tx;
        afl::charset::Utf8Charset cs;
        afl::io::ConstMemoryStream ms(OLD_FILE);
        AFL_CHECK_SUCCEEDS(a("01. loadOldFile"), game::score::Loader(tx, cs).loadOldFile(list, ms));
        AFL_CHECK_SUCCEEDS(a("02. save"), game::score::Loader(tx, cs).save(list, s));
    }

    // Minimum verification
    a.check("11. file size", s.getSize() > 8);
    a.checkEqualContent("12. file content", s.getContent().trim(7), afl::string::toBytes("CCstat0"));
    s.setPos(0);

    // Load again
    game::score::TurnScoreList list;
    {
        afl::string::NullTranslator tx;
        afl::charset::Utf8Charset cs;
        AFL_CHECK_SUCCEEDS(a("21. load"), game::score::Loader(tx, cs).load(list, s));
    }

    // Verify. Since we seeded with the same data as in testLoadOld(), this is the same test set.
    a.checkEqual("31. getNumTurns",        list.getNumTurns(), 2U);
    a.checkEqual("32. getNumScores",       list.getNumScores(), 5U);   // pla/cap/fre/bas/pbp
    a.checkEqual("33. getNumDescriptions", list.getNumDescriptions(), 0U);
    a.checkEqual("34. getTurnNumber",      list.getTurnByIndex(0)->getTurnNumber(), 3);
    a.checkEqual("35. getTurnNumber",      list.getTurnByIndex(1)->getTurnNumber(), 38);

    // Slots
    game::score::TurnScore::Slot_t planetSlot = 0, freighterSlot = 0, capitalSlot = 0, baseSlot = 0, pbpSlot = 0;
    a.check("41. planetSlot",    list.getSlot(game::score::ScoreId_Planets)    .get(planetSlot));
    a.check("42. freighterSlot", list.getSlot(game::score::ScoreId_Freighters) .get(freighterSlot));
    a.check("43. capitalSlot",   list.getSlot(game::score::ScoreId_Capital)    .get(capitalSlot));
    a.check("44. baseSlot",      list.getSlot(game::score::ScoreId_Bases)      .get(baseSlot));
    a.check("45. pbpSlot",       list.getSlot(game::score::ScoreId_BuildPoints).get(pbpSlot));

    // Verify scores
    const game::score::TurnScore* p = list.getTurn(3);
    a.checkNonNull("51. getTurn", p);
    a.checkEqual("52. planetSlot",    p->get(planetSlot,    1).orElse(-1), 5);
    a.checkEqual("53. freighterSlot", p->get(freighterSlot, 1).orElse(-1), 2);
    a.checkEqual("54. capitalSlot",   p->get(capitalSlot,   1).orElse(-1), 2);
    a.checkEqual("55. baseSlot",      p->get(baseSlot,      1).orElse(-1), 1);
    a.checkEqual("56. pbpSlot",       p->get(pbpSlot,       1).orElse(-1), 0);

    a.checkEqual("61. planetSlot",    p->get(planetSlot,    7).orElse(-1), 4);
    a.checkEqual("62. freighterSlot", p->get(freighterSlot, 7).orElse(-1), 4);
    a.checkEqual("63. capitalSlot",   p->get(capitalSlot,   7).orElse(-1), 0);
    a.checkEqual("64. baseSlot",      p->get(baseSlot,      7).orElse(-1), 1);
    a.checkEqual("65. pbpSlot",       p->get(pbpSlot,       7).orElse(-1), 0);

    p = list.getTurn(38);
    a.checkNonNull("71. getTurn", p);
    a.checkEqual("72. planetSlot",    p->get(planetSlot,    9).orElse(-1), 25);
    a.checkEqual("73. freighterSlot", p->get(freighterSlot, 9).orElse(-1), 7);
    a.checkEqual("74. capitalSlot",   p->get(capitalSlot,   9).orElse(-1), 13);
    a.checkEqual("75. baseSlot",      p->get(baseSlot,      9).orElse(-1), 4);
    a.checkEqual("76. pbpSlot",       p->get(pbpSlot,       9).orElse(-1), 1114);
}
