/**
  *  \file test/afl/io/transformreaderstreamtest.cpp
  *  \brief Test for afl::io::TransformReaderStream
  */

#include "afl/io/transformreaderstream.hpp"

#include "afl/except/fileproblemexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/filemapping.hpp"
#include "afl/io/transform.hpp"
#include "afl/test/testrunner.hpp"

using afl::base::ConstBytes_t;

namespace {
    // A sample transform.
    // Transforms input by adding one to each byte.
    // When given a flush(), writes a '!'.
    // We also want to check that flush() is called only once, and the data generated by it is requested.
    class TestTransform : public afl::io::Transform {
     public:
        TestTransform(afl::test::Assert a)
            : m_assert(a),
              m_flushPost(0),
              m_flushAck(0)
            { }
        ~TestTransform()
            {
                if (m_flushPost > 0) {
                    m_assert.checkEqual("dtor flushPost", m_flushPost, 1);
                    m_assert.checkEqual("dtor flushAck", m_flushAck, 1);
                }
            }
        void transform(afl::base::ConstBytes_t& in, afl::base::Bytes_t& out)
            {
                afl::base::Bytes_t outSave = out;
                while (!in.empty() && !out.empty()) {
                    *out.eat() = uint8_t(*in.eat() + 1);
                }
                while (m_flushPost > m_flushAck && !out.empty()) {
                    *out.eat() = '!';
                    ++m_flushAck;
                }
                out = outSave.subrange(0, outSave.size() - out.size());
            }
        void flush()
            { ++m_flushPost; }
     private:
        afl::test::Assert m_assert;
        int m_flushPost;
        int m_flushAck;
    };
}

AFL_TEST("afl.io.TransformReaderStream:read", a)
{
    // Create testee
    static const uint8_t data[] = {'h','e','l','l','o','w','o','r','l','d'};
    afl::io::ConstMemoryStream in(data);
    TestTransform tx(a);
    afl::io::TransformReaderStream testee(in, tx);

    // Read data
    uint8_t result[30];
    size_t n = testee.read(result);
    a.checkEqual("read result", n, 11U);
    a.checkEqualContent("read content", ConstBytes_t(result).trim(11), afl::string::toBytes("ifmmpxpsme!"));
}

AFL_TEST("afl.io.TransformReaderStream:methods", a)
{
    // Create testee
    static const uint8_t data[] = {'h','e','l','l','o','w','o','r','l','d'};
    afl::io::ConstMemoryStream in(data);
    TestTransform tx(a);
    afl::io::TransformReaderStream testee(in, tx);

    // Check initial state
    a.checkEqual("getPos",          testee.getPos(), afl::io::Stream::FileSize_t(0));
    a.checkEqual("getName",         testee.getName(), in.getName());
    a.checkEqual("getCapabilities", testee.getCapabilities(), afl::io::Stream::CanRead);
    a.checkEqual("getSize",         testee.getSize(), afl::io::Stream::FileSize_t(-1));

    // Must be able to seek to current position, fail to seek elsewhere
    testee.setPos(0);
    AFL_CHECK_THROWS(a("setPos to change"), testee.setPos(1), afl::except::FileProblemException);

    // Read a byte and recheck
    uint8_t tmp[1];
    testee.fullRead(tmp);
    a.checkEqual("read result", tmp[0], 'i');
    a.checkEqual("getPos after read", testee.getPos(), afl::io::Stream::FileSize_t(1));
    testee.setPos(1);
    AFL_CHECK_THROWS(a("setPos after read"), testee.setPos(10), afl::except::FileProblemException);

    // Writing fails
    AFL_CHECK_THROWS(a("write"), testee.write(tmp), afl::except::FileProblemException);

    // Null operations
    testee.createFileMapping(10);
    testee.flush();
}

/** Test position behaviour.
    There used to be a bug in TransformReaderStream to return wrong positions
    when a read operation was built from multiple transform() invocations. */
AFL_TEST("afl.io.TransformReaderStream:position", a)
{
    // A test transformation that produces single bytes.
    class Tester : public afl::io::Transform {
     public:
        virtual void transform(afl::base::ConstBytes_t& in, afl::base::Bytes_t& out)
            {
                if (in.size() > 0 && out.size() > 0) {
                    out.trim(1);
                    out.copyFrom(in.split(1));
                } else {
                    out.trim(0);
                }
            }
        virtual void flush()
            { }
    };

    // Create a stream
    Tester t;
    afl::io::ConstMemoryStream ms(afl::string::toBytes("hello"));
    afl::io::TransformReaderStream testee(ms, t);

    // Initial position must be 0
    a.checkEqual("initial position", testee.getPos(), 0U);

    // Read the content
    uint8_t data[10];
    size_t n = testee.read(data);
    a.checkEqual("read", n, 5U);

    // Final position must be 5
    a.checkEqual("getPos", testee.getPos(), 5U);
}

