/**
  *  \file test/afl/io/inflatetransformtest.cpp
  *  \brief Test for afl::io::InflateTransform
  */

#include "afl/io/inflatetransform.hpp"

#include "afl/base/countof.hpp"
#include "afl/base/types.hpp"
#include "afl/except/assertionfailedexception.hpp"
#include "afl/except/invaliddataexception.hpp"
#include "afl/except/unsupportedexception.hpp"
#include "afl/string/string.hpp"
#include "afl/test/testrunner.hpp"

using afl::base::ConstBytes_t;

namespace {
    /* file a.gz: */
    const uint8_t GOOD_GZIP_STREAM[] = {
        0x1F,0x8B,0x08,0x00,0x95,0x57,0xD2,0x4E,0x00,0x03,0x65,0x91,0x4D,0x4F,0x83,0x40,
        0x10,0x86,0xEF,0xFB,0x2B,0x26,0x98,0x98,0x85,0x56,0x68,0xAF,0x14,0x9B,0x18,0xAD,
        0x47,0xD3,0xE8,0x4D,0x6B,0x0C,0x5D,0x96,0xB2,0x11,0xD8,0x0D,0x3B,0x58,0xC5,0xF4,
        0xBF,0xCB,0x74,0x9B,0x2A,0x38,0x07,0x32,0xFB,0x3E,0xF3,0xF1,0x4E,0x88,0x82,0x80,
        0x01,0x04,0x00,0x9B,0x5C,0x95,0x12,0x50,0x5A,0xB4,0x11,0x7D,0x3B,0x65,0x42,0x21,
        0x88,0x45,0x8C,0x5D,0xA8,0x5A,0x94,0x6D,0x26,0x21,0x51,0xDA,0x62,0x23,0xD3,0x6A,
        0xF9,0xAB,0x79,0x4A,0x47,0x4E,0xB4,0x61,0xE1,0x0D,0xF5,0x7E,0x8A,0x43,0x43,0xD2,
        0xA2,0x2A,0x23,0xF9,0x29,0xA4,0x41,0x02,0x4C,0xD5,0x08,0x55,0xAA,0x6A,0x4E,0x49,
        0xDA,0xEC,0xC4,0x14,0x44,0x91,0x36,0x41,0x40,0x8F,0x0F,0x9F,0x7D,0xF7,0x3E,0x00,
        0xF6,0x05,0x59,0xE4,0x42,0xD7,0x16,0x1D,0x07,0x03,0xD7,0x10,0x4C,0x26,0xC7,0x2A,
        0x70,0x55,0x14,0xD8,0x7C,0xFD,0x79,0x51,0xAC,0xB1,0x49,0x9E,0x9C,0x73,0xC8,0xFB,
        0x26,0x97,0xC7,0xB1,0x36,0xB2,0xBE,0xEF,0xC7,0x72,0x33,0x3D,0x6B,0xB7,0x6F,0x8F,
        0xAB,0x9B,0x3B,0x7F,0x31,0x18,0x90,0x5F,0x2D,0xAD,0x94,0xEF,0x7C,0x3E,0x1B,0x81,
        0x67,0x65,0x5C,0x23,0x74,0x96,0x07,0xF9,0x88,0x92,0x4D,0xD8,0xB6,0xF9,0xCB,0x7C,
        0x36,0x7B,0x1D,0xA2,0xD3,0x39,0x74,0xB2,0xEA,0x1D,0x75,0x36,0xEC,0xA7,0x64,0xBC,
        0x2F,0x9E,0x82,0x55,0x9D,0xD4,0x39,0xE5,0xBE,0xEF,0x8F,0x4E,0xA1,0xB0,0x98,0xC5,
        0xB1,0xD0,0x2D,0x86,0xFB,0x46,0xA1,0x74,0x4D,0x6A,0xB4,0xFB,0xC0,0xFE,0x67,0x22,
        0x45,0x51,0x00,0xA7,0x93,0xD7,0x8D,0xDE,0x96,0xB2,0x5A,0x1D,0x7F,0x83,0xD2,0xF5,
        0x25,0x98,0xF1,0xAA,0xF3,0x1A,0x48,0x12,0x30,0xE1,0x4E,0x22,0x35,0x3E,0xA4,0x95,
        0xE4,0x3E,0x49,0x5E,0x0C,0x9E,0x43,0xFB,0x22,0xC5,0x93,0xB6,0xA9,0xBD,0xC5,0x68,
        0xF3,0x81,0x1D,0xD8,0x0F,0xD8,0xD4,0xDF,0x52,0x6A,0x02,0x00,0x00,
    };

    /* Decoded version of the above (a PCC2 test program): */
    const uint8_t DECODED_GZIP_STREAM[] = {
        0x2F,0x2A,0x2A,0x0A,0x20,0x20,0x2A,0x20,0x20,0x5C,0x66,0x69,0x6C,0x65,0x20,0x74,
        0x65,0x73,0x74,0x73,0x2F,0x74,0x65,0x73,0x74,0x7A,0x69,0x70,0x2E,0x63,0x63,0x0A,
        0x20,0x20,0x2A,0x2F,0x0A,0x0A,0x23,0x69,0x6E,0x63,0x6C,0x75,0x64,0x65,0x20,0x3C,
        0x69,0x6F,0x73,0x74,0x72,0x65,0x61,0x6D,0x3E,0x0A,0x23,0x69,0x6E,0x63,0x6C,0x75,
        0x64,0x65,0x20,0x22,0x69,0x6F,0x2F,0x73,0x74,0x72,0x65,0x61,0x6D,0x73,0x2E,0x68,
        0x22,0x0A,0x23,0x69,0x6E,0x63,0x6C,0x75,0x64,0x65,0x20,0x22,0x69,0x6F,0x2F,0x7A,
        0x69,0x70,0x73,0x74,0x72,0x65,0x61,0x6D,0x2E,0x68,0x22,0x0A,0x23,0x69,0x6E,0x63,
        0x6C,0x75,0x64,0x65,0x20,0x22,0x75,0x74,0x69,0x6C,0x2F,0x65,0x78,0x63,0x65,0x70,
        0x74,0x2E,0x68,0x22,0x0A,0x0A,0x69,0x6E,0x74,0x20,0x6D,0x61,0x69,0x6E,0x28,0x69,
        0x6E,0x74,0x20,0x61,0x72,0x67,0x63,0x2C,0x20,0x63,0x68,0x61,0x72,0x2A,0x2A,0x20,
        0x61,0x72,0x67,0x76,0x29,0x0A,0x7B,0x0A,0x20,0x20,0x20,0x20,0x77,0x68,0x69,0x6C,
        0x65,0x20,0x28,0x63,0x6F,0x6E,0x73,0x74,0x20,0x63,0x68,0x61,0x72,0x2A,0x20,0x70,
        0x20,0x3D,0x20,0x2A,0x2B,0x2B,0x61,0x72,0x67,0x76,0x29,0x20,0x7B,0x0A,0x20,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x74,0x72,0x79,0x20,0x7B,0x0A,0x20,0x20,0x20,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x50,0x74,0x72,0x3C,0x53,0x74,0x72,0x65,
        0x61,0x6D,0x3E,0x20,0x66,0x20,0x3D,0x20,0x53,0x74,0x72,0x65,0x61,0x6D,0x3A,0x3A,
        0x6F,0x70,0x65,0x6E,0x46,0x69,0x6C,0x65,0x28,0x70,0x2C,0x20,0x53,0x74,0x72,0x65,
        0x61,0x6D,0x3A,0x3A,0x43,0x5F,0x52,0x45,0x41,0x44,0x29,0x3B,0x0A,0x20,0x20,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x66,0x2D,0x3E,0x73,0x65,0x65,0x6B,
        0x28,0x31,0x30,0x29,0x3B,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x20,0x20,0x5A,0x69,0x70,0x53,0x74,0x72,0x65,0x61,0x6D,0x20,0x7A,0x73,0x28,0x2A,
        0x66,0x29,0x3B,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x63,0x68,0x61,0x72,0x20,0x62,0x75,0x66,0x5B,0x31,0x30,0x30,0x5D,0x3B,0x0A,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x77,0x68,0x69,0x6C,0x65,
        0x20,0x28,0x69,0x6E,0x74,0x20,0x69,0x20,0x3D,0x20,0x7A,0x73,0x2E,0x72,0x65,0x61,
        0x64,0x28,0x62,0x75,0x66,0x2C,0x20,0x73,0x69,0x7A,0x65,0x6F,0x66,0x28,0x62,0x75,
        0x66,0x29,0x29,0x29,0x20,0x7B,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x73,0x74,0x64,0x3A,0x3A,0x63,0x6F,0x75,0x74,
        0x2E,0x77,0x72,0x69,0x74,0x65,0x28,0x62,0x75,0x66,0x2C,0x20,0x69,0x29,0x3B,0x0A,
        0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x7D,0x0A,0x20,0x20,
        0x20,0x20,0x20,0x20,0x20,0x20,0x7D,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x63,0x61,0x74,0x63,0x68,0x20,0x28,0x46,0x69,0x6C,0x65,0x50,0x72,0x6F,0x62,0x6C,
        0x65,0x6D,0x45,0x78,0x63,0x65,0x70,0x74,0x69,0x6F,0x6E,0x26,0x20,0x70,0x29,0x20,
        0x7B,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x73,0x74,
        0x64,0x3A,0x3A,0x63,0x6F,0x75,0x74,0x20,0x3C,0x3C,0x20,0x70,0x2E,0x67,0x65,0x74,
        0x46,0x69,0x6C,0x65,0x4E,0x61,0x6D,0x65,0x28,0x29,0x20,0x3C,0x3C,0x20,0x22,0x3A,
        0x20,0x22,0x20,0x3C,0x3C,0x20,0x70,0x2E,0x77,0x68,0x61,0x74,0x28,0x29,0x20,0x3C,
        0x3C,0x20,0x22,0x5C,0x6E,0x22,0x3B,0x0A,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x7D,0x0A,0x20,0x20,0x20,0x20,0x7D,0x0A,0x7D,0x0A,
    };
    const size_t DECODED_GZIP_STREAM_size = sizeof(DECODED_GZIP_STREAM);

    /* file a.damaged.gz: */
    const uint8_t BAD_GZIP_STREAM[] = {
        0x1F,0x8B,0x08,0x00,0x95,0x57,0xD2,0x4E,0x00,0x03,0x65,0x91,0x4D,0x4F,0x83,0x40,
        0x10,0x86,0xEF,0xFB,0x2B,0x26,0x98,0x98,0x85,0x56,0x68,0xAF,0x14,0x9B,0x18,0xAD,
        0x47,0xD3,0xE8,0x4D,0x6B,0x43,0x43,0x62,0x73,0x69,0x6D,0x32,0x1A,0x01,0x80,0x53,
        0x68,0x69,0x70,0x20,0x31,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
        0x20,0x20,0x20,0x00,0x00,0x3A,0x00,0x01,0x00,0x0C,0x00,0x00,0x00,0x0A,0x00,0x01,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x00,0x01,0x00,0x64,0x00,0x3F,
        0x3F,0x3F,0xFF,0xFF,0x14,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x01,0x00,0x0B,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x01,0x00,0x0A,0x00,0x0A,0x00,0x64,0x00,0x3F,0x3F,0x3F,0xFF,0xFF,0x00,0x00,0x00,
        0x00,0x0A,
    };

    static const size_t blockSizes[] = { 1, 2, 3, 4, 10, 100, 255, 256, 1024 };
}

/** Test a good stream. Test that it decodes correctly even when read
    with different input sizes. */
AFL_TEST("afl.io.InflateTransform:good", a)
{
    if (!afl::io::InflateTransform::isAvailable()) {
        // Test how it refuses being constructed.
        AFL_CHECK_THROWS(a("InflateTransform not supported"), new afl::io::InflateTransform(afl::io::InflateTransform::Gzip),
                         afl::except::UnsupportedException);
        return;
    }

    /*
     *  Regular test
     */
    for (size_t i = 0; i < countof(blockSizes); ++i) {
        // Create it
        afl::io::InflateTransform inf(afl::io::InflateTransform::Gzip);
        afl::base::ConstBytes_t data(GOOD_GZIP_STREAM);

        // Post it through the InflateTransform in pieces
        String_t result;
        uint8_t buffer[4096];
        for (size_t pos = 0; pos < data.size(); pos += blockSizes[i]) {
            afl::base::ConstBytes_t range(data.subrange(pos, blockSizes[i]));
            while (!range.empty()) {
                afl::base::Bytes_t out(buffer);
                inf.transform(range, out);
                result += afl::string::fromBytes(out);
            }
        }

        // Check result
        a.checkEqual("result size", result.size(), DECODED_GZIP_STREAM_size);
        a.checkEqualContent("result content", afl::string::toBytes(result), ConstBytes_t(DECODED_GZIP_STREAM));
    }
}

/** Test a good stream. Test that it decodes correctly even when read
    with different output sizes. */
AFL_TEST("afl.io.InflateTransform:good2", a)
{
    if (!afl::io::InflateTransform::isAvailable()) {
        // Test how it refuses being constructed.
        AFL_CHECK_THROWS(a("InflateTransform not supported"), new afl::io::InflateTransform(afl::io::InflateTransform::Gzip),
                         afl::except::UnsupportedException);
        return;
    }

    /*
     *  Regular test
     */
    for (size_t i = 0; i < countof(blockSizes); ++i) {
        // Create it
        afl::io::InflateTransform inf(afl::io::InflateTransform::Gzip);
        afl::base::ConstBytes_t data(GOOD_GZIP_STREAM);

        // Post it through the InflateTransform, fetching output in pieces
        String_t result;
        uint8_t buffer[4096];
        while (!data.empty()) {
            afl::base::Bytes_t out(afl::base::Bytes_t(buffer).trim(blockSizes[i]));
            inf.transform(data, out);
            result += afl::string::fromBytes(out);
        }

        // Check result
        a.checkEqual("result szie", result.size(), DECODED_GZIP_STREAM_size);
        a.checkEqualContent("result content", afl::string::toBytes(result), ConstBytes_t(DECODED_GZIP_STREAM));
    }
}

/** Test a bad stream. We cannot test when exactly the failure
    happens, but it must be detected before the end of the stream is
    reached. */
AFL_TEST("afl.io.InflateTransform:bad", a)
{
    if (!afl::io::InflateTransform::isAvailable()) {
        return;
    }

    /*
     *  Regular test
     */
    for (size_t i = 0; i < countof(blockSizes); ++i) {
        // Create it
        afl::io::InflateTransform inf(afl::io::InflateTransform::Gzip);
        afl::base::ConstBytes_t data(BAD_GZIP_STREAM);

        // Post it through the InflateTransform in pieces
        try {
            uint8_t buffer[4096];
            for (size_t pos = 0; pos < data.size(); pos += blockSizes[i]) {
                afl::base::ConstBytes_t range(data.subrange(pos, blockSizes[i]));
                while (!range.empty()) {
                    afl::base::Bytes_t out(buffer);
                    inf.transform(range, out);
                }
            }

            // Must not get here
            a.fail("must not reach");
        }
        catch (afl::except::AssertionFailedException& ex) {
            throw;
        }
        catch (afl::except::InvalidDataException& ide) {
            // OK
        }
        catch (...) {
            a.check("must not reach 2", 0);
        }
    }
}

/** Test truncation. */
AFL_TEST("afl.io.InflateTransform:truncated", a)
{
    if (!afl::io::InflateTransform::isAvailable()) {
        return;
    }

    // Test data
    static const uint8_t HI[] = {
        0x1f, 0x8b, 0x08, 0x08, 0x46, 0x62, 0xc9, 0x58, 0x02, 0x03, 0x68, 0x69,
        0x00, 0xcb, 0xc8, 0x04, 0x00, 0xac, 0x2a, 0x93, 0xd8, 0x02, 0x00, 0x00,
        0x00
    };

    // Starting at any place other than the first byte will fail
    for (size_t i = 1; i < sizeof(HI) - 10; ++i) {
        afl::base::ConstBytes_t in(HI);
        in.split(i);

        uint8_t buffer[4096];
        afl::base::Bytes_t out(buffer);

        afl::io::InflateTransform inf(afl::io::InflateTransform::Gzip);
        AFL_CHECK_THROWS(a("removed beginning"), inf.transform(in, out), afl::except::InvalidDataException);
    }

    // Ending before the stream ends will not fail, but demonstrate that we can shut down in any state
    for (size_t i = 0; i < sizeof(HI); ++i) {
        afl::base::ConstBytes_t in(HI);
        in.trim(i);

        uint8_t buffer[4096];
        afl::base::Bytes_t out(buffer);

        afl::io::InflateTransform inf(afl::io::InflateTransform::Gzip);
        AFL_CHECK_SUCCEEDS(a("removed end"), inf.transform(in, out));
    }
}

/** Test header variant. */
AFL_TEST("afl.io.InflateTransform:header-variant", a)
{
    if (!afl::io::InflateTransform::isAvailable()) {
        return;
    }

    // Data.
    // I cannot find a tool that creates the extra header fields, so this is manually crafted (gunzip decompresses it ok).
    static const uint8_t HI[] = {
        0x1f, 0x8b, 0x08, 0x1C,     // header
        0x46, 0x62, 0xc9, 0x58,     // mtime
        0x02, 0x03,                 // flag/os
        7, 0, 5,5,5,5,5,5,5,        // extra data
        0x68, 0x69, 0x00,           // filename
        1,1,1,1,1,1,1,1, 0,         // comment
        0xcb, 0xc8, 0x04, 0x00, 0xac, 0x2a, 0x93, 0xd8, 0x02, 0x00, 0x00, 0x00
    };

    // Decompress
    afl::base::ConstBytes_t in(HI);
    uint8_t buffer[4096];
    afl::base::Bytes_t out(buffer);
    afl::io::InflateTransform(afl::io::InflateTransform::Gzip).transform(in, out);

    a.checkEqual("size", out.size(), 2U);
    a.checkEqual("byte 0", buffer[0], 'h');
    a.checkEqual("byte 1", buffer[1], 'i');
}
