/**
  *  \file test/afl/io/archive/arreadertest.cpp
  *  \brief Test for afl::io::archive::ArReader
  */

#include "afl/io/archive/arreader.hpp"

#include <cstring>
#include "afl/except/fileformatexception.hpp"
#include "afl/except/fileproblemexception.hpp"
#include "afl/io/constmemorystream.hpp"
#include "afl/io/directoryentry.hpp"
#include "afl/io/stream.hpp"
#include "afl/string/posixfilenames.hpp"
#include "afl/test/testrunner.hpp"

using afl::base::Enumerator;
using afl::base::Ptr;
using afl::base::Ref;
using afl::io::DirectoryEntry;

namespace {
    /* Simple file containing two text files:
          one.txt ("abcdefg")
          two.txt ("ABCDEFG") */
    const uint8_t SIMPLE_FILE[] = {
        0x21, 0x3c, 0x61, 0x72, 0x63, 0x68, 0x3e, 0x0a, 0x6f, 0x6e, 0x65, 0x2e,
        0x74, 0x78, 0x74, 0x2f, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x31, 0x35, 0x38, 0x38, 0x31, 0x37, 0x31, 0x38, 0x38, 0x31, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x20, 0x38, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x61, 0x62, 0x63, 0x64,
        0x65, 0x66, 0x67, 0x0a, 0x74, 0x77, 0x6f, 0x2e, 0x74, 0x78, 0x74, 0x2f,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x35, 0x38, 0x38,
        0x31, 0x37, 0x31, 0x38, 0x38, 0x36, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30,
        0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x36,
        0x34, 0x34, 0x20, 0x20, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x60, 0x0a, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x0a
    };

    /* A little more complex file.
          file_with_a_really_long_name.txt ("ABCDEFGH")
          short.sh ("hi"; executable) */
    const uint8_t OTHER_FILE[] = {
        0x21, 0x3c, 0x61, 0x72, 0x63, 0x68, 0x3e, 0x0a, 0x2f, 0x2f, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x34, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x66, 0x69, 0x6c, 0x65,
        0x5f, 0x77, 0x69, 0x74, 0x68, 0x5f, 0x61, 0x5f, 0x72, 0x65, 0x61, 0x6c,
        0x6c, 0x79, 0x5f, 0x6c, 0x6f, 0x6e, 0x67, 0x5f, 0x6e, 0x61, 0x6d, 0x65,
        0x2e, 0x74, 0x78, 0x74, 0x2f, 0x0a, 0x2f, 0x30, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x35,
        0x38, 0x38, 0x31, 0x37, 0x33, 0x34, 0x37, 0x33, 0x20, 0x20, 0x31, 0x30,
        0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30,
        0x30, 0x36, 0x34, 0x34, 0x20, 0x20, 0x39, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
        0x47, 0x48, 0x0a, 0x0a, 0x73, 0x68, 0x6f, 0x72, 0x74, 0x2e, 0x73, 0x68,
        0x2f, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x35, 0x38, 0x38,
        0x31, 0x37, 0x33, 0x34, 0x38, 0x37, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30,
        0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x37,
        0x35, 0x35, 0x20, 0x20, 0x33, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x60, 0x0a, 0x68, 0x69, 0x0a, 0x0a
    };

    const uint8_t LIB_FILE[] = {
        0x21, 0x3c, 0x61, 0x72, 0x63, 0x68, 0x3e, 0x0a, 0x2f, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x31, 0x35, 0x38, 0x38, 0x31, 0x37, 0x35, 0x31, 0x30, 0x37, 0x20, 0x20,
        0x30, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x30, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x32, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x50, 0x66, 0x6f, 0x6f, 0x00, 0x74, 0x2e, 0x6f, 0x2f,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x31, 0x35, 0x38, 0x38, 0x31, 0x37, 0x35, 0x30, 0x39, 0x39, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x20, 0x31, 0x32, 0x31, 0x36,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x7f, 0x45, 0x4c, 0x46,
        0x02, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x3e, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x0b, 0x00, 0x08, 0x00,
        0x55, 0x48, 0x89, 0xe5, 0xb8, 0x2a, 0x00, 0x00, 0x00, 0x5d, 0xc3, 0x00,
        0x47, 0x43, 0x43, 0x3a, 0x20, 0x28, 0x44, 0x65, 0x62, 0x69, 0x61, 0x6e,
        0x20, 0x34, 0x2e, 0x39, 0x2e, 0x32, 0x2d, 0x31, 0x30, 0x29, 0x20, 0x34,
        0x2e, 0x39, 0x2e, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x7a, 0x52, 0x00,
        0x01, 0x78, 0x10, 0x01, 0x1b, 0x0c, 0x07, 0x08, 0x90, 0x01, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x00, 0x00, 0x00, 0x41, 0x0e, 0x10, 0x86, 0x02, 0x43, 0x0d,
        0x06, 0x46, 0x0c, 0x07, 0x08, 0x00, 0x00, 0x00, 0x00, 0x2e, 0x73, 0x79,
        0x6d, 0x74, 0x61, 0x62, 0x00, 0x2e, 0x73, 0x74, 0x72, 0x74, 0x61, 0x62,
        0x00, 0x2e, 0x73, 0x68, 0x73, 0x74, 0x72, 0x74, 0x61, 0x62, 0x00, 0x2e,
        0x74, 0x65, 0x78, 0x74, 0x00, 0x2e, 0x64, 0x61, 0x74, 0x61, 0x00, 0x2e,
        0x62, 0x73, 0x73, 0x00, 0x2e, 0x63, 0x6f, 0x6d, 0x6d, 0x65, 0x6e, 0x74,
        0x00, 0x2e, 0x6e, 0x6f, 0x74, 0x65, 0x2e, 0x47, 0x4e, 0x55, 0x2d, 0x73,
        0x74, 0x61, 0x63, 0x6b, 0x00, 0x2e, 0x72, 0x65, 0x6c, 0x61, 0x2e, 0x65,
        0x68, 0x5f, 0x66, 0x72, 0x61, 0x6d, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0xf1, 0xff, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x74, 0x2e, 0x63, 0x00, 0x66, 0x6f, 0x6f, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x21, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x4b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x35, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x11, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    /* Erroneous file. Same as SIMPLE_FILE, but with bad header on second file. */
    const uint8_t BAD_FILE[] = {
        0x21, 0x3c, 0x61, 0x72, 0x63, 0x68, 0x3e, 0x0a, 0x6f, 0x6e, 0x65, 0x2e,
        0x74, 0x78, 0x74, 0x2f, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x31, 0x35, 0x38, 0x38, 0x31, 0x37, 0x31, 0x38, 0x38, 0x31, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20,
        0x31, 0x30, 0x30, 0x36, 0x34, 0x34, 0x20, 0x20, 0x38, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x60, 0x0a, 0x61, 0x62, 0x63, 0x64,
        0x65, 0x66, 0x67, 0x0a, 0x74, 0x77, 0x6f, 0x2e, 0x74, 0x78, 0x74, 0x2f,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x35, 0x38, 0x38,
        0x31, 0x37, 0x31, 0x38, 0x38, 0x36, 0x20, 0x20, 0x31, 0x30, 0x30, 0x30,
        0x20, 0x20, 0x31, 0x30, 0x30, 0x30, 0x20, 0x20, 0x31, 0x30, 0x30, 0x36,
        0x34, 0x34, 0x20, 0x20, 0x38, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x33, 0x44, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x0a
    };
}

/** Test listing a file.
    The call must produce the known members in correct order. */
AFL_TEST("afl.io.archive.ArReader:getDirectoryEntries", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(SIMPLE_FILE));

    Ref<Enumerator<Ptr<DirectoryEntry> > > iter = testee->getDirectoryEntries();
    Ptr<DirectoryEntry> e;

    a.check("01. getNextElement", iter->getNextElement(e));
    a.check("02. get", e.get() != 0);
    a.checkEqual("03. getTitle", e->getTitle(), "one.txt");
    a.check("04. getFileType", e->getFileType() == DirectoryEntry::tFile);
    a.check("05. getFlags", e->getFlags().empty());

    a.check("11. getNextElement", iter->getNextElement(e));
    a.check("12. get", e.get() != 0);
    a.checkEqual("13. getTitle", e->getTitle(), "two.txt");
    a.check("14. getFileType", e->getFileType() == DirectoryEntry::tFile);
    a.check("15. getFlags", e->getFlags().empty());

    a.check("21. getNextElement", !iter->getNextElement(e));

    // Coverage
    AFL_CHECK_SUCCEEDS(a("31. flush"), testee->flush());
}

/** Test listing a file.
    The call must produce the known members in correct order.
    This tests long file names and the 'executable' bit. */
AFL_TEST("afl.io.archive.ArReader:getDirectoryEntries:special-files", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(OTHER_FILE));

    Ref<Enumerator<Ptr<DirectoryEntry> > > iter = testee->getDirectoryEntries();
    Ptr<DirectoryEntry> e;

    a.check("01. getNextElement", iter->getNextElement(e));
    a.check("02. get", e.get() != 0);
    a.checkEqual("03. getTitle", e->getTitle(), "file_with_a_really_long_name.txt");
    a.check("04. getFileType", e->getFileType() == DirectoryEntry::tFile);
    a.check("05. getFlags", e->getFlags().empty());

    uint8_t buffer[100];
    size_t n = e->openFile(afl::io::FileSystem::OpenRead)->read(buffer);
    a.checkEqual("11. read", n, 9U);
    a.checkEqual("12. content", std::memcmp(buffer, "ABCDEFGH\n", 9), 0);

    a.check("21. getNextElement", iter->getNextElement(e));
    a.check("22. get", e.get() != 0);
    a.checkEqual("23. getTitle", e->getTitle(), "short.sh");
    a.check("24. getFileType", e->getFileType() == DirectoryEntry::tFile);
    a.check("25. getFlags", e->getFlags() == DirectoryEntry::FileFlags_t(DirectoryEntry::Executable));

    a.check("31. getNextElement", !iter->getNextElement(e));
}

/** Test listing a file.
    The call must produce the known members in correct order.
    This tests skipping special members. */
AFL_TEST("afl.io.archive.ArReader:getDirectoryEntries:library-file", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(LIB_FILE));

    Ref<Enumerator<Ptr<DirectoryEntry> > > iter = testee->getDirectoryEntries();
    Ptr<DirectoryEntry> e;

    a.check("01. getNextElement", iter->getNextElement(e));
    a.check("02. get", e.get() != 0);
    a.checkEqual("03. getTitle", e->getTitle(), "t.o");
    a.check("04. getFileType", e->getFileType() == DirectoryEntry::tFile);
    a.check("05. getFlags", e->getFlags().empty());

    a.check("11. getNextElement", !iter->getNextElement(e));
}

/** Test reading a file.
    Performs random, by-name access.
    Individual streams must be usable and not affect each other. */
AFL_TEST("afl.io.archive.ArReader:read", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(SIMPLE_FILE));

    uint8_t buf[1];
    Ref<afl::io::Stream> one = testee->openFile("one.txt", afl::io::FileSystem::OpenRead);
    one->fullRead(buf);
    a.checkEqual("01. read", buf[0], 'a');
    a.checkEqual("02. getSize", one->getSize(), 8U);
    a.checkEqual("03. getPos", one->getPos(), 1U);

    Ref<afl::io::Stream> two = testee->openFile("two.txt", afl::io::FileSystem::OpenRead);
    two->fullRead(buf);
    a.checkEqual("11. read", buf[0], 'A');

    Ref<afl::io::Stream> oneAgain = testee->openFile("one.txt", afl::io::FileSystem::OpenRead);
    oneAgain->fullRead(buf);
    a.checkEqual("21. read", buf[0], 'a');

    one->fullRead(buf);
    a.checkEqual("31. read", buf[0], 'b');
    two->fullRead(buf);
    a.checkEqual("32. read", buf[0], 'B');
    oneAgain->fullRead(buf);
    a.checkEqual("33. read", buf[0], 'b');
}

/** Test reading a file, with createChild(). */
AFL_TEST("afl.io.archive.ArReader:read:createChild", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(SIMPLE_FILE));

    uint8_t buf[1];
    Ref<afl::io::Stream> one = testee->openFile("one.txt", afl::io::FileSystem::OpenRead);
    Ref<afl::io::Stream> two = one->createChild(0);
    Ref<afl::io::Stream> three = two->createChild(0);

    one->fullRead(buf);
    a.checkEqual("01. name", afl::string::PosixFileNames().getFileName(one->getName()), "one.txt");
    a.checkEqual("02. read", buf[0], 'a');
    a.checkEqual("03. getSize", one->getSize(), 8U);
    a.checkEqual("04. getPos", one->getPos(), 1U);

    two->fullRead(buf);
    a.checkEqual("11. name", afl::string::PosixFileNames().getFileName(two->getName()), "one.txt");
    a.checkEqual("12. read", buf[0], 'a');
    a.checkEqual("13. getSize", two->getSize(), 8U);
    a.checkEqual("14. getPos", two->getPos(), 1U);

    three->fullRead(buf);
    a.checkEqual("21. name", afl::string::PosixFileNames().getFileName(three->getName()), "one.txt");
    a.checkEqual("22. read", buf[0], 'a');
    a.checkEqual("23. getSize", three->getSize(), 8U);
    a.checkEqual("24. getPos", three->getPos(), 1U);
}

/** Test some failures. */
AFL_TEST("afl.io.archive.ArReader:failure", a)
{
    // Disallowed accesses on valid file
    static const uint8_t x[] = {1,2,3,4,5,6,7,8,9,10};
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(SIMPLE_FILE));

    // - open nonexistant
    AFL_CHECK_THROWS(a("open nonexistant"), testee->openFile("notfound", afl::io::FileSystem::OpenRead), afl::except::FileProblemException);

    // - create anew
    AFL_CHECK_THROWS(a("create new"), testee->openFile("new", afl::io::FileSystem::Create), afl::except::FileProblemException);

    // - erase
    AFL_CHECK_THROWS(a("erase"), testee->erase("one.txt"), afl::except::FileProblemException);

    // - write
    AFL_CHECK_THROWS(a("write"), testee->openFile("one.txt", afl::io::FileSystem::OpenRead)->write(x), afl::except::FileProblemException);

    // - open as directory
    AFL_CHECK_THROWS(a("openDirectory"), testee->getDirectoryEntryByName("one.txt")->openDirectory(), afl::except::FileProblemException);

    // Invalid files
    AFL_CHECK_THROWS(a("open empty"), afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(afl::base::Nothing)), afl::except::FileFormatException);
    AFL_CHECK_THROWS(a("open bogus"), afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(x)), afl::except::FileFormatException);
}

/** Test invalid file header. */
AFL_TEST("afl.io.archive.ArReader:error:header", a)
{
    Ref<afl::io::archive::ArReader> testee = afl::io::archive::ArReader::open(*new afl::io::ConstMemoryStream(BAD_FILE));

    // Can open one, but not two.
    AFL_CHECK_SUCCEEDS(a("open one"), testee->openFile("one.txt", afl::io::FileSystem::OpenRead));
    AFL_CHECK_THROWS(a("open two"), testee->openFile("two.txt", afl::io::FileSystem::OpenRead), afl::except::FileProblemException);

    // Can open one again (error is not sticky)
    AFL_CHECK_SUCCEEDS(a("open one"), testee->openFile("one.txt", afl::io::FileSystem::OpenRead));
}
