/**
  *  \file test/afl/checksums/sha512test.cpp
  *  \brief Test for afl::checksums::SHA512
  */

#include "afl/checksums/sha512.hpp"
#include "afl/test/testrunner.hpp"

#include "afl/base/countof.hpp"

using afl::base::ConstBytes_t;

namespace {
    struct TestCase {
        const char* m_string;
        uint32_t m_repeat;
        const uint8_t m_expect[64];
    };
}

/** Basic tests. */
AFL_TEST("afl.checksums.SHA512", a)
{
    afl::checksums::SHA512 cc;
    uint8_t bytes[64];

    // Querying an empty hash should return the expected result
    static const uint8_t expected[] = {
        0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd, 0xf1, 0x54,
        0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07, 0xd6, 0x20, 0xe4, 0x05,
        0x0b, 0x57, 0x15, 0xdc, 0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c,
        0xe9, 0xce, 0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
        0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f, 0x63, 0xb9,
        0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81, 0xa5, 0x38, 0x32, 0x7a,
        0xf9, 0x27, 0xda, 0x3e,
    };
    ConstBytes_t result = cc.getHash(bytes);
    a.check("getHash empty 1", result.equalContent(expected));

    // Asking again should get the same result again
    result = cc.getHash(bytes);
    a.check("getHash empty 2", result.equalContent(expected));

    // Check string
    a.checkEqual("getHashAsHexString", cc.getHashAsHexString(), "cf83e1357eefb8bdf1542850d66d8007d620e4050b5715dc83f4a921d36ce9ce47d0d13c5d85f2b0ff8318d2877eec2f63b931bd47417a81a538327af927da3e");

    // Asking a smaller buffer, must get correct content (first half of hash)
    uint8_t small[8];
    result = cc.getHash(small);
    a.checkEqual("getHash small", result.size(), 8U);
    a.checkEqualContent("getHash small content", ConstBytes_t(bytes).trim(8), ConstBytes_t(small));

    // Asking a larger buffer, must get correct content (just the hash)
    uint8_t large[100];
    result = cc.getHash(large);
    a.checkEqual("getHash large", result.size(), 64U);
    a.checkEqualContent("getHash large content", ConstBytes_t(bytes), ConstBytes_t(large).trim(64));

    // Self-information
    a.checkEqual("getHashSize 1", cc.getHashSize(), 64U);
    a.checkEqual("getHashSize 2", cc.getHashSize(), cc.HASH_SIZE);
    a.checkEqual("getBlockSize", cc.getBlockSize(), 128U);
}

/** Hash tests from RFC 6234. */
AFL_TEST("afl.checksums.SHA512:getHash:rfc6234", a)
{
    static const TestCase cases[] = {
        { "abc", 1,
          { 0xDD, 0xAF, 0x35, 0xA1, 0x93, 0x61, 0x7A, 0xBA, 0xCC, 0x41, 0x73, 0x49, 0xAE, 0x20, 0x41, 0x31, 0x12, 0xE6, 0xFA, 0x4E, 0x89, 0xA9, 0x7E, 0xA2,
            0x0A, 0x9E, 0xEE, 0xE6, 0x4B, 0x55, 0xD3, 0x9A, 0x21, 0x92, 0x99, 0x2A, 0x27, 0x4F, 0xC1, 0xA8, 0x36, 0xBA, 0x3C, 0x23, 0xA3, 0xFE, 0xEB, 0xBD,
            0x45, 0x4D, 0x44, 0x23, 0x64, 0x3C, 0xE8, 0x0E, 0x2A, 0x9A, 0xC9, 0x4F, 0xA5, 0x4C, 0xA4, 0x9F } },
        { "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
          "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu", 1,
          { 0x8E, 0x95, 0x9B, 0x75, 0xDA, 0xE3, 0x13, 0xDA, 0x8C, 0xF4, 0xF7, 0x28, 0x14, 0xFC, 0x14, 0x3F, 0x8F, 0x77, 0x79, 0xC6, 0xEB, 0x9F, 0x7F, 0xA1,
            0x72, 0x99, 0xAE, 0xAD, 0xB6, 0x88, 0x90, 0x18, 0x50, 0x1D, 0x28, 0x9E, 0x49, 0x00, 0xF7, 0xE4, 0x33, 0x1B, 0x99, 0xDE, 0xC4, 0xB5, 0x43, 0x3A,
            0xC7, 0xD3, 0x29, 0xEE, 0xB6, 0xDD, 0x26, 0x54, 0x5E, 0x96, 0xE5, 0x5B, 0x87, 0x4B, 0xE9, 0x09 } },
        { "a", 1000000,
          { 0xE7, 0x18, 0x48, 0x3D, 0x0C, 0xE7, 0x69, 0x64, 0x4E, 0x2E, 0x42, 0xC7, 0xBC, 0x15, 0xB4, 0x63, 0x8E, 0x1F, 0x98, 0xB1, 0x3B, 0x20, 0x44, 0x28,
            0x56, 0x32, 0xA8, 0x03, 0xAF, 0xA9, 0x73, 0xEB, 0xDE, 0x0F, 0xF2, 0x44, 0x87, 0x7E, 0xA6, 0x0A, 0x4C, 0xB0, 0x43, 0x2C, 0xE5, 0x77, 0xC3, 0x1B,
            0xEB, 0x00, 0x9C, 0x5C, 0x2C, 0x49, 0xAA, 0x2E, 0x4E, 0xAD, 0xB2, 0x17, 0xAD, 0x8C, 0xC0, 0x9B } },
        { "0123456701234567012345670123456701234567012345670123456701234567", 10,
          { 0x89, 0xD0, 0x5B, 0xA6, 0x32, 0xC6, 0x99, 0xC3, 0x12, 0x31, 0xDE, 0xD4, 0xFF, 0xC1, 0x27, 0xD5, 0xA8, 0x94, 0xDA, 0xD4, 0x12, 0xC0, 0xE0, 0x24,
            0xDB, 0x87, 0x2D, 0x1A, 0xBD, 0x2B, 0xA8, 0x14, 0x1A, 0x0F, 0x85, 0x07, 0x2A, 0x9B, 0xE1, 0xE2, 0xAA, 0x04, 0xCF, 0x33, 0xC7, 0x65, 0xCB, 0x51,
            0x08, 0x13, 0xA3, 0x9C, 0xD5, 0xA8, 0x4C, 0x4A, 0xCA, 0xA6, 0x4D, 0x3F, 0x3F, 0xB7, 0xBA, 0xE9 } },
        { "\xD0", 1,
          { 0x99, 0x92, 0x20, 0x29, 0x38, 0xE8, 0x82, 0xE7, 0x3E, 0x20, 0xF6, 0xB6, 0x9E, 0x68, 0xA0, 0xA7, 0x14, 0x90, 0x90, 0x42, 0x3D, 0x93, 0xC8, 0x1B,
            0xAB, 0x3F, 0x21, 0x67, 0x8D, 0x4A, 0xCE, 0xEE, 0xE5, 0x0E, 0x4E, 0x8C, 0xAF, 0xAD, 0xA4, 0xC8, 0x5A, 0x54, 0xEA, 0x83, 0x06, 0x82, 0x6C, 0x4A,
            0xD6, 0xE7, 0x4C, 0xEC, 0xE9, 0x63, 0x1B, 0xFA, 0x8A, 0x54, 0x9B, 0x4A, 0xB3, 0xFB, 0xBA, 0x15 } },
        { "\x8d\x4e\x3c\x0e\x38\x89\x19\x14\x91\x81\x6e\x9d\x98\xbf\xf0\xa0", 1,
          { 0xCB, 0x0B, 0x67, 0xA4, 0xB8, 0x71, 0x2C, 0xD7, 0x3C, 0x9A, 0xAB, 0xC0, 0xB1, 0x99, 0xE9, 0x26, 0x9B, 0x20, 0x84, 0x4A, 0xFB, 0x75, 0xAC, 0xBD,
            0xD1, 0xC1, 0x53, 0xC9, 0x82, 0x89, 0x24, 0xC3, 0xDD, 0xED, 0xAA, 0xFE, 0x66, 0x9C, 0x5F, 0xDD, 0x0B, 0xC6, 0x6F, 0x63, 0x0F, 0x67, 0x73, 0x98,
            0x82, 0x13, 0xEB, 0x1B, 0x16, 0xF5, 0x17, 0xAD, 0x0D, 0xE4, 0xB2, 0xF0, 0xC9, 0x5C, 0x90, 0xF8 } },
    };

    for (size_t i = 0; i < countof(cases); ++i) {
        // Compute the hash
        afl::checksums::SHA512 cc;
        for (size_t j = 0; j < cases[i].m_repeat; ++j) {
            cc.add(afl::string::toBytes(cases[i].m_string));
        }

        uint8_t buffer[64];
        a.check("", cc.getHash(buffer).equalContent(cases[i].m_expect));
    }
}

/** HMAC tests from RFC 6234. */
AFL_TEST("afl.checksums.SHA512:computeHMAC:rfc6234", a)
{
    afl::checksums::SHA512 cc;

    // 1
    cc.computeHMAC(afl::string::toBytes("\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"),
                   afl::string::toBytes("\x48\x69\x20\x54\x68\x65\x72\x65"));
    a.checkEqual("case 1", cc.getHashAsHexString(), "87aa7cdea5ef619d4ff0b4241a1d6cb02379f4e2ce4ec2787ad0b30545e17cdedaa833b7d6b8a702038b274eaea3f4e4be9d914eeb61f1702e696c203a126854");

    // 2
    cc.computeHMAC(afl::string::toBytes("\x4a\x65\x66\x65"),
                   afl::string::toBytes("\x77\x68\x61\x74\x20\x64\x6f\x20\x79\x61\x20\x77\x61\x6e\x74"
                                        "\x20\x66\x6f\x72\x20\x6e\x6f\x74\x68\x69\x6e\x67\x3f"));
    a.checkEqual("case 2", cc.getHashAsHexString(), "164b7a7bfcf819e2e395fbe73b56e0a387bd64222e831fd610270cd7ea2505549758bf75c05a994a6d034f65f8f0e6fdcaeab1a34d4a6b4b636e070a38bce737");

    // 3
    cc.computeHMAC(afl::string::toBytes("\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
                                        "\xaa\xaa\xaa\xaa\xaa"),
                   afl::string::toBytes("\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
                                        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
                                        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
                                        "\xdd\xdd\xdd\xdd\xdd"));
    a.checkEqual("case 3", cc.getHashAsHexString(), "fa73b0089d56a284efb0f0756c890be9b1b5dbdd8ee81a3655f83e33b2279d39bf3e848279a722c806b485a47e67c807b946a337bee8942674278859e13292fb");

    // 4
    cc.computeHMAC(afl::string::toBytes("\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
                                        "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19"),
                   afl::string::toBytes("\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
                                        "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
                                        "\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd\xcd"
                                        "\xcd\xcd\xcd\xcd\xcd"));
    a.checkEqual("case 4", cc.getHashAsHexString(), "b0ba465637458c6990e5a8c5f61d4af7e576d97ff94b872de76f8050361ee3dba91ca5c11aa25eb4d679275cc5788063a5f19741120c4f2de2adebeb10a298dd");
}
