/**
  *  \file afl/test/callreceiver.hpp
  *  \brief Class afl::test::CallReceiver
  */
#ifndef AFL_AFL_TEST_CALLRECEIVER_HPP
#define AFL_AFL_TEST_CALLRECEIVER_HPP

#include <list>
#include "afl/test/assert.hpp"
#include "afl/string/string.hpp"
#include "afl/base/deletable.hpp"
#include "afl/container/ptrvector.hpp"

namespace afl { namespace test {

    /** Generic function call mock.
        This can be used to implement a mock for a given function call interface.

        The user configures a sequence of calls and return values using expectCall(), provideReturnValue().
        The mock is implemented using checkCall() and consumeReturnValue().
        CallReceiver verifies that the two sequences match.
        There is no restriction about the number of calls or return values processed per call,
        e.g. if your function returns a tuple, you can consume two return values.

        A function call is identified by a string.
        This string is typically generated by stringifying the arguments, for example using afl::string::Format().
        There are no rules how this string looks like and what arguments it encodes, but also no further help.

        Return values must match exactly, a provideReturnValue(String_t) must be matched with a consumeReturnValue<String_t>;
        there are no automatic conversions.

        To implement a mock, derive from the original interface and from CallReceiver, and implement the interface methods.
        Example:
        <code>
           class MyMock : public afl::test::CallReceiver, public MyInterface {
            public:
               MyMock(const Assert& a)
                   : CallReceiver(a), MyInterface()
                   { }
               virtual int myFunction(String_t s)
                   {
                       checkCall(afl::string::Format("myFunction(%s)", s));
                       return consumeReturnValue<int>();
                   }
           };
        </code> */
    class CallReceiver {
     public:
        /** Constructor.
            \param a Assertion generator */
        explicit CallReceiver(const Assert& a);

        /** Destructor. */
        ~CallReceiver();

        /** Expect a function call (call from test).
            \param call Stringified call. */
        void expectCall(String_t call);

        /** Verify a function call (call from mock).
            \param call Stringified call
            \throw afl::except::AssertionFailedException if call is not expected */
        void checkCall(String_t call);

        /** Verify that call sequence is complete (call from test).
            \throw afl::except::AssertionFailedException if not all expected calls have been made, all return values have been consumed */
        void checkFinish();

        /** Provide a return value (call from test).
            \tparam T type
            \param t value */
        template<typename T>
        void provideReturnValue(T t);

        /** Consume a return value (call from mock).
            \tparam T type
            \return value */
        template<typename T>
        T& consumeReturnValue();

     private:
        Assert m_assert;
        std::list<String_t> m_queue;

        template<typename T>
        class Value : public afl::base::Deletable {
         public:
            Value(T t)
                : m_t(t)
                { }
            T& get()
                { return m_t; }
         private:
            T m_t;
        };

        afl::container::PtrVector<afl::base::Deletable> m_returnValues;
        size_t m_nextReturnValue;
    };


} }

template<typename T>
void
afl::test::CallReceiver::provideReturnValue(T t)
{
    m_returnValues.pushBackNew(new Value<T>(t));
}

template<typename T>
T&
afl::test::CallReceiver::consumeReturnValue()
{
    m_assert.check("consumeReturnValue: must have another result", m_nextReturnValue < m_returnValues.size());
    Value<T>* p = dynamic_cast<Value<T>*>(m_returnValues[m_nextReturnValue++]);
    m_assert.check("consumeReturnValue: must have correct type", p != 0);
    return p->get();
}

#endif
