/**
  *  \file afl/charset/unicode.cpp
  *  \brief Unicode Definitions
  */

#include "afl/charset/unicode.hpp"
#include "afl/base/memory.hpp"

namespace {
    /*
     *  Case mapping
     *
     *  Case mapping is performed using tables with optimized storage (below 1 kB per table).
     *  Case pairs typically are arranged either as a sequence of upper-case letters and a sequence of lower-case letters,
     *  or as two alternating sequences.
     *  Therefore, it suffices to store for each mapping
     *  - the first character of the input letter
     *  - the first character of the output letter
     *  - the offset between characters in the sequence
     *  - the number of characters in the sequence
     *
     *  For simplicity, we make a table for upper-case and one for lower-case conversion.
     *  This also allows us to express asymmetric mappings, such as U+01C4..01C6 (DZ/Dz/dz).
     *  On the other hand, I limited the table to characters below U+3000;
     *  adding more gains little but adds some special cases that would enlargen the table sizes.
     *  Therefore, these tables cover WGL4 and then some, but not all of Unicode.
     *
     *  Tables have been generated from UnicodeData.txt and therefore use a language-neutral mapping.
     */

    struct Mapping {
        uint16_t input;         // Input character
        uint16_t output;        // Output character
        uint8_t offset;         // Offset, must be power of two
        uint8_t count;          // Number of characters
    };

    const Mapping UPPERCASE_TABLE[] = {
        { 0x0061, 0x0041,    1,   26 },
        { 0x00B5, 0x039C,    1,    1 },
        { 0x00E0, 0x00C0,    1,   23 },
        { 0x00F8, 0x00D8,    1,    7 },
        { 0x00FF, 0x0178,    1,    1 },
        { 0x0101, 0x0100,    2,   24 },
        { 0x0131, 0x0049,    1,    1 },
        { 0x0133, 0x0132,    2,    3 },
        { 0x013A, 0x0139,    2,    8 },
        { 0x014B, 0x014A,    2,   23 },
        { 0x017A, 0x0179,    2,    3 },
        { 0x017F, 0x0053,    1,    1 },
        { 0x0180, 0x0243,    1,    1 },
        { 0x0183, 0x0182,    2,    2 },
        { 0x0188, 0x0187,    4,    2 },
        { 0x0192, 0x0191,    1,    1 },
        { 0x0195, 0x01F6,    1,    1 },
        { 0x0199, 0x0198,    1,    1 },
        { 0x019A, 0x023D,    1,    1 },
        { 0x019E, 0x0220,    1,    1 },
        { 0x01A1, 0x01A0,    2,    3 },
        { 0x01A8, 0x01A7,    1,    1 },
        { 0x01AD, 0x01AC,    1,    1 },
        { 0x01B0, 0x01AF,    4,    2 },
        { 0x01B6, 0x01B5,    1,    1 },
        { 0x01B9, 0x01B8,    4,    2 },
        { 0x01BF, 0x01F7,    1,    1 },
        { 0x01C5, 0x01C4,    1,    1 },
        { 0x01C6, 0x01C4,    1,    1 },
        { 0x01C8, 0x01C7,    1,    1 },
        { 0x01C9, 0x01C7,    1,    1 },
        { 0x01CB, 0x01CA,    1,    1 },
        { 0x01CC, 0x01CA,    1,    1 },
        { 0x01CE, 0x01CD,    2,    8 },
        { 0x01DD, 0x018E,    1,    1 },
        { 0x01DF, 0x01DE,    2,    9 },
        { 0x01F2, 0x01F1,    1,    1 },
        { 0x01F3, 0x01F1,    1,    1 },
        { 0x01F5, 0x01F4,    4,    2 },
        { 0x01FB, 0x01FA,    2,   19 },
        { 0x0223, 0x0222,    2,    9 },
        { 0x023C, 0x023B,    1,    1 },
        { 0x023F, 0x2C7E,    1,    2 },
        { 0x0242, 0x0241,    1,    1 },
        { 0x0247, 0x0246,    2,    5 },
        { 0x0250, 0x2C6F,    1,    1 },
        { 0x0251, 0x2C6D,    1,    1 },
        { 0x0252, 0x2C70,    1,    1 },
        { 0x0253, 0x0181,    1,    1 },
        { 0x0254, 0x0186,    1,    1 },
        { 0x0256, 0x0189,    1,    2 },
        { 0x0259, 0x018F,    1,    1 },
        { 0x025B, 0x0190,    1,    1 },
        { 0x0260, 0x0193,    1,    1 },
        { 0x0263, 0x0194,    1,    1 },
        { 0x0268, 0x0197,    1,    1 },
        { 0x0269, 0x0196,    1,    1 },
        { 0x026B, 0x2C62,    1,    1 },
        { 0x026F, 0x019C,    1,    1 },
        { 0x0271, 0x2C6E,    1,    1 },
        { 0x0272, 0x019D,    1,    1 },
        { 0x0275, 0x019F,    1,    1 },
        { 0x027D, 0x2C64,    1,    1 },
        { 0x0280, 0x01A6,    1,    1 },
        { 0x0283, 0x01A9,    1,    1 },
        { 0x0288, 0x01AE,    1,    1 },
        { 0x0289, 0x0244,    1,    1 },
        { 0x028A, 0x01B1,    1,    2 },
        { 0x028C, 0x0245,    1,    1 },
        { 0x0292, 0x01B7,    1,    1 },
        { 0x0345, 0x0399,    1,    1 },
        { 0x0371, 0x0370,    2,    2 },
        { 0x0377, 0x0376,    1,    1 },
        { 0x037B, 0x03FD,    1,    3 },
        { 0x03AC, 0x0386,    1,    1 },
        { 0x03AD, 0x0388,    1,    3 },
        { 0x03B1, 0x0391,    1,   17 },
        { 0x03C2, 0x03A3,    1,    1 },
        { 0x03C3, 0x03A3,    1,    9 },
        { 0x03CC, 0x038C,    1,    1 },
        { 0x03CD, 0x038E,    1,    2 },
        { 0x03D0, 0x0392,    1,    1 },
        { 0x03D1, 0x0398,    1,    1 },
        { 0x03D5, 0x03A6,    1,    1 },
        { 0x03D6, 0x03A0,    1,    1 },
        { 0x03D7, 0x03CF,    1,    1 },
        { 0x03D9, 0x03D8,    2,   12 },
        { 0x03F0, 0x039A,    1,    1 },
        { 0x03F1, 0x03A1,    1,    1 },
        { 0x03F2, 0x03F9,    1,    1 },
        { 0x03F5, 0x0395,    1,    1 },
        { 0x03F8, 0x03F7,    1,    1 },
        { 0x03FB, 0x03FA,    1,    1 },
        { 0x0430, 0x0410,    1,   32 },
        { 0x0450, 0x0400,    1,   16 },
        { 0x0461, 0x0460,    2,   17 },
        { 0x048B, 0x048A,    2,   27 },
        { 0x04C2, 0x04C1,    2,    7 },
        { 0x04CF, 0x04C0,    1,    1 },
        { 0x04D1, 0x04D0,    2,   44 },
        { 0x0561, 0x0531,    1,   38 },
        { 0x1D7D, 0x2C63,    1,    1 },
        { 0x1E01, 0x1E00,    2,   75 },
        { 0x1E9B, 0x1E60,    1,    1 },
        { 0x1EA1, 0x1EA0,    2,   48 },
        { 0x1F00, 0x1F08,    1,    8 },
        { 0x1F10, 0x1F18,    1,    6 },
        { 0x1F20, 0x1F28,    1,    8 },
        { 0x1F30, 0x1F38,    1,    8 },
        { 0x1F40, 0x1F48,    1,    6 },
        { 0x1F51, 0x1F59,    2,    4 },
        { 0x1F60, 0x1F68,    1,    8 },
        { 0x1F70, 0x1FBA,    1,    2 },
        { 0x1F72, 0x1FC8,    1,    4 },
        { 0x1F76, 0x1FDA,    1,    2 },
        { 0x1F78, 0x1FF8,    1,    2 },
        { 0x1F7A, 0x1FEA,    1,    2 },
        { 0x1F7C, 0x1FFA,    1,    2 },
        { 0x1F80, 0x1F88,    1,    8 },
        { 0x1F90, 0x1F98,    1,    8 },
        { 0x1FA0, 0x1FA8,    1,    8 },
        { 0x1FB0, 0x1FB8,    1,    2 },
        { 0x1FB3, 0x1FBC,    1,    1 },
        { 0x1FBE, 0x0399,    1,    1 },
        { 0x1FC3, 0x1FCC,    1,    1 },
        { 0x1FD0, 0x1FD8,    1,    2 },
        { 0x1FE0, 0x1FE8,    1,    2 },
        { 0x1FE5, 0x1FEC,    1,    1 },
        { 0x1FF3, 0x1FFC,    1,    1 },
        { 0x214E, 0x2132,    1,    1 },
        { 0x2170, 0x2160,    1,   16 },
        { 0x2184, 0x2183,    1,    1 },
        { 0x24D0, 0x24B6,    1,   26 },
        { 0x2C30, 0x2C00,    1,   47 },
        { 0x2C61, 0x2C60,    1,    1 },
        { 0x2C65, 0x023A,    1,    1 },
        { 0x2C66, 0x023E,    1,    1 },
        { 0x2C68, 0x2C67,    2,    3 },
        { 0x2C73, 0x2C72,    1,    1 },
        { 0x2C76, 0x2C75,    1,    1 },
        { 0x2C81, 0x2C80,    2,   50 },
        { 0x2CEC, 0x2CEB,    2,    2 },
        { 0x2D00, 0x10A0,    1,   38 },
    };

    const Mapping LOWERCASE_TABLE[] = {
        { 0x0041, 0x0061,    1,   26 },
        { 0x00C0, 0x00E0,    1,   23 },
        { 0x00D8, 0x00F8,    1,    7 },
        { 0x0100, 0x0101,    2,   24 },
        { 0x0130, 0x0069,    1,    1 },
        { 0x0132, 0x0133,    2,    3 },
        { 0x0139, 0x013A,    2,    8 },
        { 0x014A, 0x014B,    2,   23 },
        { 0x0178, 0x00FF,    1,    1 },
        { 0x0179, 0x017A,    2,    3 },
        { 0x0181, 0x0253,    1,    1 },
        { 0x0182, 0x0183,    2,    2 },
        { 0x0186, 0x0254,    1,    1 },
        { 0x0187, 0x0188,    1,    1 },
        { 0x0189, 0x0256,    1,    2 },
        { 0x018B, 0x018C,    1,    1 },
        { 0x018E, 0x01DD,    1,    1 },
        { 0x018F, 0x0259,    1,    1 },
        { 0x0190, 0x025B,    1,    1 },
        { 0x0191, 0x0192,    1,    1 },
        { 0x0193, 0x0260,    1,    1 },
        { 0x0194, 0x0263,    1,    1 },
        { 0x0196, 0x0269,    1,    1 },
        { 0x0197, 0x0268,    1,    1 },
        { 0x0198, 0x0199,    1,    1 },
        { 0x019C, 0x026F,    1,    1 },
        { 0x019D, 0x0272,    1,    1 },
        { 0x019F, 0x0275,    1,    1 },
        { 0x01A0, 0x01A1,    2,    3 },
        { 0x01A6, 0x0280,    1,    1 },
        { 0x01A7, 0x01A8,    1,    1 },
        { 0x01A9, 0x0283,    1,    1 },
        { 0x01AC, 0x01AD,    1,    1 },
        { 0x01AE, 0x0288,    1,    1 },
        { 0x01AF, 0x01B0,    1,    1 },
        { 0x01B1, 0x028A,    1,    2 },
        { 0x01B3, 0x01B4,    2,    2 },
        { 0x01B7, 0x0292,    1,    1 },
        { 0x01B8, 0x01B9,    4,    2 },
        { 0x01C4, 0x01C6,    1,    1 },
        { 0x01C5, 0x01C6,    1,    1 },
        { 0x01C7, 0x01C9,    1,    1 },
        { 0x01C8, 0x01C9,    1,    1 },
        { 0x01CA, 0x01CC,    1,    1 },
        { 0x01CB, 0x01CC,    2,    9 },
        { 0x01DE, 0x01DF,    2,    9 },
        { 0x01F1, 0x01F3,    1,    1 },
        { 0x01F2, 0x01F3,    2,    2 },
        { 0x01F6, 0x0195,    1,    1 },
        { 0x01F7, 0x01BF,    1,    1 },
        { 0x01F8, 0x01F9,    2,   20 },
        { 0x0220, 0x019E,    1,    1 },
        { 0x0222, 0x0223,    2,    9 },
        { 0x023A, 0x2C65,    1,    1 },
        { 0x023B, 0x023C,    1,    1 },
        { 0x023D, 0x019A,    1,    1 },
        { 0x023E, 0x2C66,    1,    1 },
        { 0x0241, 0x0242,    1,    1 },
        { 0x0243, 0x0180,    1,    1 },
        { 0x0244, 0x0289,    1,    1 },
        { 0x0245, 0x028C,    1,    1 },
        { 0x0246, 0x0247,    2,    5 },
        { 0x0370, 0x0371,    2,    2 },
        { 0x0376, 0x0377,    1,    1 },
        { 0x0386, 0x03AC,    1,    1 },
        { 0x0388, 0x03AD,    1,    3 },
        { 0x038C, 0x03CC,    1,    1 },
        { 0x038E, 0x03CD,    1,    2 },
        { 0x0391, 0x03B1,    1,   17 },
        { 0x03A3, 0x03C3,    1,    9 },
        { 0x03CF, 0x03D7,    1,    1 },
        { 0x03D8, 0x03D9,    2,   12 },
        { 0x03F4, 0x03B8,    1,    1 },
        { 0x03F7, 0x03F8,    1,    1 },
        { 0x03F9, 0x03F2,    1,    1 },
        { 0x03FA, 0x03FB,    1,    1 },
        { 0x03FD, 0x037B,    1,    3 },
        { 0x0400, 0x0450,    1,   16 },
        { 0x0410, 0x0430,    1,   32 },
        { 0x0460, 0x0461,    2,   17 },
        { 0x048A, 0x048B,    2,   27 },
        { 0x04C0, 0x04CF,    1,    1 },
        { 0x04C1, 0x04C2,    2,    7 },
        { 0x04D0, 0x04D1,    2,   44 },
        { 0x0531, 0x0561,    1,   38 },
        { 0x10A0, 0x2D00,    1,   38 },
        { 0x1E00, 0x1E01,    2,   75 },
        { 0x1E9E, 0x00DF,    1,    1 },
        { 0x1EA0, 0x1EA1,    2,   48 },
        { 0x1F08, 0x1F00,    1,    8 },
        { 0x1F18, 0x1F10,    1,    6 },
        { 0x1F28, 0x1F20,    1,    8 },
        { 0x1F38, 0x1F30,    1,    8 },
        { 0x1F48, 0x1F40,    1,    6 },
        { 0x1F59, 0x1F51,    2,    4 },
        { 0x1F68, 0x1F60,    1,    8 },
        { 0x1F88, 0x1F80,    1,    8 },
        { 0x1F98, 0x1F90,    1,    8 },
        { 0x1FA8, 0x1FA0,    1,    8 },
        { 0x1FB8, 0x1FB0,    1,    2 },
        { 0x1FBA, 0x1F70,    1,    2 },
        { 0x1FBC, 0x1FB3,    1,    1 },
        { 0x1FC8, 0x1F72,    1,    4 },
        { 0x1FCC, 0x1FC3,    1,    1 },
        { 0x1FD8, 0x1FD0,    1,    2 },
        { 0x1FDA, 0x1F76,    1,    2 },
        { 0x1FE8, 0x1FE0,    1,    2 },
        { 0x1FEA, 0x1F7A,    1,    2 },
        { 0x1FEC, 0x1FE5,    1,    1 },
        { 0x1FF8, 0x1F78,    1,    2 },
        { 0x1FFA, 0x1F7C,    1,    2 },
        { 0x1FFC, 0x1FF3,    1,    1 },
        { 0x2126, 0x03C9,    1,    1 },
        { 0x212A, 0x006B,    1,    1 },
        { 0x212B, 0x00E5,    1,    1 },
        { 0x2132, 0x214E,    1,    1 },
        { 0x2160, 0x2170,    1,   16 },
        { 0x2183, 0x2184,    1,    1 },
        { 0x24B6, 0x24D0,    1,   26 },
        { 0x2C00, 0x2C30,    1,   47 },
        { 0x2C60, 0x2C61,    1,    1 },
        { 0x2C62, 0x026B,    1,    1 },
        { 0x2C63, 0x1D7D,    1,    1 },
        { 0x2C64, 0x027D,    1,    1 },
        { 0x2C67, 0x2C68,    2,    3 },
        { 0x2C6D, 0x0251,    1,    1 },
        { 0x2C6E, 0x0271,    1,    1 },
        { 0x2C6F, 0x0250,    1,    1 },
        { 0x2C70, 0x0252,    1,    1 },
        { 0x2C72, 0x2C73,    1,    1 },
        { 0x2C75, 0x2C76,    1,    1 },
        { 0x2C7E, 0x023F,    1,    2 },
        { 0x2C80, 0x2C81,    2,   50 },
        { 0x2CEB, 0x2CEC,    2,    2 },
    };

    afl::charset::Unichar_t lookup(afl::charset::Unichar_t ch, afl::base::Memory<const Mapping> tab)
    {
        while (const Mapping* p = tab.eat()) {
            if (ch >= p->input
                && ch < afl::charset::Unichar_t(p->input + p->offset*p->count)
                && ((ch - p->input) & (p->offset-1)) == 0)
            {
                return ch + p->output - p->input;
            }
        }
        return ch;
    }
}

afl::charset::Unichar_t
afl::charset::getUpperCase(Unichar_t ch)
{
    return lookup(ch, UPPERCASE_TABLE);
}

afl::charset::Unichar_t
afl::charset::getLowerCase(Unichar_t ch)
{
    return lookup(ch, LOWERCASE_TABLE);
}
